-- StaySpot Enhanced Database Schema
-- Updated with partnerships, chat, and transportation features
-- Version: 3.0.0

SET FOREIGN_KEY_CHECKS = 0;

-- Drop new tables first
DROP TABLE IF EXISTS 
    driver_locations, driver_bookings, transportation_vehicles, drivers,
    chat_messages, chat_rooms,
    partnership_applications,
    vendor_reviews, vendor_documents, vendor_assignments, vendor_service_categories, vendors,
    inspection_items, inspections,
    maintenance_photos, maintenance_notes, maintenance_requests,
    rent_payments, expenses,
    messages, notifications,
    leads, insurance_policies,
    property_documents, property_images,
    leases,
    audit_logs, user_sessions, auth_attempts, system_settings,
    properties,
    users,
    -- Enum tables
    service_categories, payment_statuses, lease_statuses, maintenance_statuses, 
    maintenance_priorities, property_statuses, property_types, user_roles;

SET FOREIGN_KEY_CHECKS = 1;

-- Create database
CREATE DATABASE IF NOT EXISTS stayspotco_stayspot;
USE stayspotco_stayspot;

-- ==================== ENUMERATION TABLES ====================

CREATE TABLE user_roles (
    role_code VARCHAR(50) PRIMARY KEY,
    role_name VARCHAR(100) NOT NULL,
    description TEXT,
    permissions JSON,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

CREATE TABLE property_types (
    type_code VARCHAR(50) PRIMARY KEY,
    type_name VARCHAR(100) NOT NULL,
    description TEXT,
    icon_class VARCHAR(100),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

CREATE TABLE property_statuses (
    status_code VARCHAR(50) PRIMARY KEY,
    status_name VARCHAR(100) NOT NULL,
    description TEXT,
    color_class VARCHAR(50),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

CREATE TABLE maintenance_priorities (
    priority_code VARCHAR(20) PRIMARY KEY,
    priority_name VARCHAR(50) NOT NULL,
    description TEXT,
    sla_hours INT,
    color_class VARCHAR(50),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

CREATE TABLE maintenance_statuses (
    status_code VARCHAR(50) PRIMARY KEY,
    status_name VARCHAR(100) NOT NULL,
    description TEXT,
    color_class VARCHAR(50),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

CREATE TABLE lease_statuses (
    status_code VARCHAR(50) PRIMARY KEY,
    status_name VARCHAR(100) NOT NULL,
    description TEXT,
    color_class VARCHAR(50),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

CREATE TABLE payment_statuses (
    status_code VARCHAR(50) PRIMARY KEY,
    status_name VARCHAR(100) NOT NULL,
    description TEXT,
    color_class VARCHAR(50),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

CREATE TABLE service_categories (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    category_code VARCHAR(50) NOT NULL UNIQUE,
    category_name VARCHAR(100) NOT NULL,
    description TEXT,
    parent_category_code VARCHAR(50) NULL,
    icon_class VARCHAR(100),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    
    INDEX idx_parent_category (parent_category_code),
    FOREIGN KEY (parent_category_code) REFERENCES service_categories(category_code)
);

-- ==================== CORE USER TABLES ====================

CREATE TABLE users (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    uuid CHAR(36) NOT NULL UNIQUE DEFAULT (UUID()),
    
    -- Basic Information
    first_name VARCHAR(50) NOT NULL,
    last_name VARCHAR(50) NOT NULL,
    email VARCHAR(255) NOT NULL UNIQUE,
    phone VARCHAR(20),
    password_hash VARCHAR(255) NOT NULL,
    
    -- Role Management
    role_code VARCHAR(50) NOT NULL,
    
    -- Profile Information
    avatar_url VARCHAR(500),
    bio TEXT,
    date_of_birth DATE,
    
    -- Address Information
    address_street VARCHAR(255),
    address_city VARCHAR(100),
    address_state VARCHAR(100),
    address_zip_code VARCHAR(20),
    address_country VARCHAR(100) DEFAULT 'KE',
    latitude DECIMAL(10,8),
    longitude DECIMAL(11,8),
    
    -- Company/Professional Information
    company_name VARCHAR(255),
    company_position VARCHAR(100),
    license_number VARCHAR(100),
    tax_id VARCHAR(100),
    
    -- Emergency Contact
    emergency_contact_name VARCHAR(100),
    emergency_contact_phone VARCHAR(20),
    emergency_contact_relationship VARCHAR(50),
    
    -- Preferences & Settings (JSON)
    preferences JSON,
    settings JSON,
    
    -- Security & Status
    is_email_verified BOOLEAN DEFAULT FALSE,
    is_phone_verified BOOLEAN DEFAULT FALSE,
    is_active BOOLEAN DEFAULT TRUE,
    last_login DATETIME,
    
    -- Two-Factor Authentication
    two_factor_enabled BOOLEAN DEFAULT FALSE,
    two_factor_secret VARCHAR(255),
    
    -- Password Management
    password_reset_token VARCHAR(255),
    password_reset_expires DATETIME,
    
    -- Soft Delete
    deleted_at DATETIME NULL,
    
    -- Audit Fields
    created_by BIGINT NULL,
    updated_by BIGINT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    -- Indexes
    INDEX idx_email (email),
    INDEX idx_role (role_code),
    INDEX idx_active (is_active),
    INDEX idx_company (company_name),
    INDEX idx_created_at (created_at),
    INDEX idx_deleted (deleted_at),
    INDEX idx_name (first_name, last_name),
    
    -- Foreign Keys
    FOREIGN KEY (role_code) REFERENCES user_roles(role_code),
    FOREIGN KEY (created_by) REFERENCES users(id),
    FOREIGN KEY (updated_by) REFERENCES users(id)
);

-- ==================== PROPERTY MANAGEMENT ====================

CREATE TABLE properties (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    uuid CHAR(36) NOT NULL UNIQUE DEFAULT (UUID()),
    
    -- Basic Information
    title VARCHAR(255) NOT NULL,
    description TEXT,
    property_type_code VARCHAR(50) NOT NULL,
    status_code VARCHAR(50) NOT NULL,
    
    -- Location Details
    address_street VARCHAR(255) NOT NULL,
    address_city VARCHAR(100) NOT NULL,
    address_state VARCHAR(100) NOT NULL,
    address_zip_code VARCHAR(20) NOT NULL,
    address_country VARCHAR(100) DEFAULT 'KE',
    latitude DECIMAL(10,8),
    longitude DECIMAL(11,8),
    
    -- Property Specifications
    bedrooms INT NOT NULL DEFAULT 0,
    bathrooms DECIMAL(3,1) NOT NULL DEFAULT 0,
    area_sqft INT,
    year_built YEAR,
    lot_size_sqft INT,
    
    -- Rental Information
    monthly_rent DECIMAL(10,2) NOT NULL,
    security_deposit DECIMAL(10,2),
    available_from DATE,
    is_published BOOLEAN DEFAULT FALSE,
    
    -- Features & Amenities (JSON for flexibility)
    amenities JSON,
    interior_features JSON,
    kitchen_amenities JSON,
    bathroom_features JSON,
    outdoor_amenities JSON,
    smart_home_features JSON,
    accessibility_features JSON,
    utility_information JSON,
    rules JSON,
    features JSON,
    
    -- Media
    main_image_url VARCHAR(500),
    images JSON,
    
    -- Ownership & Management
    owner_id BIGINT NOT NULL,
    property_manager_id BIGINT NULL,
    primary_agent_id BIGINT NULL,
    
    -- Soft Delete
    deleted_at DATETIME NULL,
    
    -- Audit Fields
    created_by BIGINT NOT NULL,
    updated_by BIGINT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    -- Indexes
    INDEX idx_owner (owner_id),
    INDEX idx_manager (property_manager_id),
    INDEX idx_agent (primary_agent_id),
    INDEX idx_location (address_city, address_state),
    INDEX idx_rent (monthly_rent),
    INDEX idx_bedrooms (bedrooms),
    INDEX idx_status (status_code),
    INDEX idx_available (available_from),
    INDEX idx_deleted (deleted_at),
    INDEX idx_published (is_published),
    FULLTEXT idx_search (title, description, address_street),
    
    -- Foreign Keys
    FOREIGN KEY (property_type_code) REFERENCES property_types(type_code),
    FOREIGN KEY (status_code) REFERENCES property_statuses(status_code),
    FOREIGN KEY (owner_id) REFERENCES users(id),
    FOREIGN KEY (property_manager_id) REFERENCES users(id),
    FOREIGN KEY (primary_agent_id) REFERENCES users(id),
    FOREIGN KEY (created_by) REFERENCES users(id),
    FOREIGN KEY (updated_by) REFERENCES users(id)
);

-- ==================== LEASE MANAGEMENT ====================

CREATE TABLE leases (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    uuid CHAR(36) NOT NULL UNIQUE DEFAULT (UUID()),
    
    -- Lease Information
    property_id BIGINT NOT NULL,
    tenant_id BIGINT NOT NULL,
    landlord_id BIGINT NOT NULL,
    lease_number VARCHAR(100) NOT NULL UNIQUE,
    
    -- Term Details
    start_date DATE NOT NULL,
    end_date DATE NOT NULL,
    monthly_rent DECIMAL(10,2) NOT NULL,
    security_deposit DECIMAL(10,2) NOT NULL,
    
    -- Occupants
    primary_tenant_id BIGINT NOT NULL,
    occupants_count INT DEFAULT 1,
    occupants_details JSON,
    
    -- Lease Status
    status_code VARCHAR(50) NOT NULL DEFAULT 'pending',
    
    -- Documents
    lease_document_url VARCHAR(500),
    
    -- Financial
    rent_due_day INT DEFAULT 1,
    late_fee_percentage DECIMAL(5,2) DEFAULT 5.0,
    grace_period_days INT DEFAULT 5,
    
    -- Special Terms
    special_terms TEXT,
    terms JSON,
    
    -- Dates
    signed_at DATETIME,
    terminated_at DATETIME,
    
    -- Soft Delete
    deleted_at DATETIME NULL,
    
    -- Audit Fields
    created_by BIGINT NOT NULL,
    updated_by BIGINT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    -- Indexes
    INDEX idx_property (property_id),
    INDEX idx_tenant (tenant_id),
    INDEX idx_landlord (landlord_id),
    INDEX idx_primary_tenant (primary_tenant_id),
    INDEX idx_status (status_code),
    INDEX idx_dates (start_date, end_date),
    INDEX idx_lease_number (lease_number),
    INDEX idx_deleted (deleted_at),
    
    -- Foreign Keys
    FOREIGN KEY (property_id) REFERENCES properties(id),
    FOREIGN KEY (tenant_id) REFERENCES users(id),
    FOREIGN KEY (landlord_id) REFERENCES users(id),
    FOREIGN KEY (primary_tenant_id) REFERENCES users(id),
    FOREIGN KEY (status_code) REFERENCES lease_statuses(status_code),
    FOREIGN KEY (created_by) REFERENCES users(id),
    FOREIGN KEY (updated_by) REFERENCES users(id)
);

-- ==================== PAYMENT MANAGEMENT ====================

CREATE TABLE payments (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    uuid CHAR(36) NOT NULL UNIQUE DEFAULT (UUID()),
    
    -- Payment Details
    lease_id BIGINT NOT NULL,
    tenant_id BIGINT NOT NULL,
    landlord_id BIGINT NOT NULL,
    property_id BIGINT NOT NULL,
    
    -- Amount Information
    amount_due DECIMAL(10,2) NOT NULL,
    amount_paid DECIMAL(10,2) DEFAULT 0,
    late_fee_amount DECIMAL(10,2) DEFAULT 0,
    
    -- Period Information
    period_month INT NOT NULL,
    period_year INT NOT NULL,
    due_date DATE NOT NULL,
    paid_date DATETIME,
    
    -- Payment Method
    payment_method ENUM('credit_card', 'debit_card', 'bank_transfer', 'cash', 'check', 'online', 'paypal', 'stripe', 'mpesa'),
    transaction_id VARCHAR(255),
    payment_gateway VARCHAR(100),
    receipt_url VARCHAR(500),
    
    -- Status
    status_code VARCHAR(50) NOT NULL DEFAULT 'pending',
    
    -- Description
    description TEXT,
    
    -- Audit Fields
    created_by BIGINT NOT NULL,
    updated_by BIGINT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    -- Indexes
    INDEX idx_lease (lease_id),
    INDEX idx_tenant (tenant_id),
    INDEX idx_landlord (landlord_id),
    INDEX idx_property (property_id),
    INDEX idx_period (period_year, period_month),
    INDEX idx_due_date (due_date),
    INDEX idx_paid_date (paid_date),
    INDEX idx_status (status_code),
    INDEX idx_transaction (transaction_id),
    
    -- Foreign Keys
    FOREIGN KEY (lease_id) REFERENCES leases(id),
    FOREIGN KEY (tenant_id) REFERENCES users(id),
    FOREIGN KEY (landlord_id) REFERENCES users(id),
    FOREIGN KEY (property_id) REFERENCES properties(id),
    FOREIGN KEY (status_code) REFERENCES payment_statuses(status_code),
    FOREIGN KEY (created_by) REFERENCES users(id),
    FOREIGN KEY (updated_by) REFERENCES users(id)
);

-- ==================== MAINTENANCE MANAGEMENT ====================

CREATE TABLE maintenance_requests (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    uuid CHAR(36) NOT NULL UNIQUE DEFAULT (UUID()),
    
    -- Request Details
    property_id BIGINT NOT NULL,
    unit_id VARCHAR(100),
    requested_by BIGINT NOT NULL,
    assigned_to BIGINT NULL,
    landlord_id BIGINT NOT NULL,
    
    -- Issue Information
    title VARCHAR(255) NOT NULL,
    description TEXT NOT NULL,
    category ENUM('plumbing', 'electrical', 'hvac', 'appliance', 'structural', 'cosmetic', 'pest_control', 'other') NOT NULL DEFAULT 'other',
    priority_code VARCHAR(20) NOT NULL DEFAULT 'normal',
    
    -- Status & Tracking
    status_code VARCHAR(50) NOT NULL DEFAULT 'submitted',
    estimated_cost DECIMAL(10,2),
    actual_cost DECIMAL(10,2),
    rating INT,
    
    -- Media
    images JSON,
    
    -- Scheduling
    requested_date DATETIME DEFAULT CURRENT_TIMESTAMP,
    scheduled_date DATETIME,
    completed_date DATETIME,
    
    -- Location Access
    access_instructions TEXT,
    tenant_available BOOLEAN DEFAULT FALSE,
    notes TEXT,
    
    -- Audit Fields
    created_by BIGINT NOT NULL,
    updated_by BIGINT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    -- Indexes
    INDEX idx_property (property_id),
    INDEX idx_requested_by (requested_by),
    INDEX idx_assigned_to (assigned_to),
    INDEX idx_landlord (landlord_id),
    INDEX idx_priority (priority_code),
    INDEX idx_status (status_code),
    INDEX idx_category (category),
    INDEX idx_requested_date (requested_date),
    INDEX idx_scheduled_date (scheduled_date),
    
    -- Foreign Keys
    FOREIGN KEY (property_id) REFERENCES properties(id),
    FOREIGN KEY (requested_by) REFERENCES users(id),
    FOREIGN KEY (assigned_to) REFERENCES users(id),
    FOREIGN KEY (landlord_id) REFERENCES users(id),
    FOREIGN KEY (priority_code) REFERENCES maintenance_priorities(priority_code),
    FOREIGN KEY (status_code) REFERENCES maintenance_statuses(status_code),
    FOREIGN KEY (created_by) REFERENCES users(id),
    FOREIGN KEY (updated_by) REFERENCES users(id)
);

-- ==================== NOTIFICATION & MESSAGING ====================

CREATE TABLE notifications (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    uuid CHAR(36) NOT NULL UNIQUE DEFAULT (UUID()),
    
    -- Notification Details
    user_id BIGINT NOT NULL,
    title VARCHAR(255) NOT NULL,
    message TEXT NOT NULL,
    type ENUM(
        'payment_reminder',
        'maintenance_update', 
        'lease_expiry',
        'new_message',
        'system_alert',
        'booking_confirmation',
        'payment_received',
        'maintenance_scheduled',
        'partnership_update',
        'driver_request'
    ) NOT NULL,
    
    -- Context
    entity_type ENUM('property', 'lease', 'payment', 'maintenance', 'user', 'partnership', 'booking'),
    entity_id BIGINT,
    action_url VARCHAR(500),
    metadata JSON,
    
    -- Status
    is_read BOOLEAN DEFAULT FALSE,
    read_at DATETIME,
    priority ENUM('low', 'medium', 'high') DEFAULT 'medium',
    
    -- Delivery
    send_email BOOLEAN DEFAULT FALSE,
    send_sms BOOLEAN DEFAULT FALSE,
    email_sent BOOLEAN DEFAULT FALSE,
    sms_sent BOOLEAN DEFAULT FALSE,
    
    -- Expiry
    expires_at DATETIME,
    
    -- Audit Fields
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    -- Indexes
    INDEX idx_user (user_id),
    INDEX idx_type (type),
    INDEX idx_entity (entity_type, entity_id),
    INDEX idx_created (created_at),
    INDEX idx_read (is_read),
    INDEX idx_priority (priority),
    
    -- Foreign Keys
    FOREIGN KEY (user_id) REFERENCES users(id)
);

-- ==================== PARTNERSHIP SYSTEM ====================

CREATE TABLE partnership_applications (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    uuid CHAR(36) NOT NULL UNIQUE DEFAULT (UUID()),
    
    -- Company Information
    company_name VARCHAR(255) NOT NULL,
    company_type VARCHAR(100) NOT NULL,
    industry VARCHAR(100) NOT NULL,
    website VARCHAR(255),
    
    -- Contact Information
    contact_name VARCHAR(100) NOT NULL,
    contact_email VARCHAR(255) NOT NULL,
    contact_phone VARCHAR(20),
    contact_position VARCHAR(100),
    
    -- Company Details
    company_size VARCHAR(50),
    annual_revenue VARCHAR(50),
    years_in_business INT,
    
    -- Address
    address_street VARCHAR(255),
    address_city VARCHAR(100),
    address_state VARCHAR(100),
    address_zip_code VARCHAR(20),
    address_country VARCHAR(100) DEFAULT 'KE',
    
    -- Partnership Details
    partnership_type VARCHAR(100) NOT NULL,
    services_offered TEXT,
    target_market TEXT,
    partnership_goals TEXT,
    
    -- Additional Information
    previous_partnerships TEXT,
    company_references TEXT,
    additional_info TEXT,
    
    -- Status
    status ENUM('pending', 'under_review', 'approved', 'rejected', 'on_hold') DEFAULT 'pending',
    reviewed_by BIGINT NULL,
    reviewed_at DATETIME NULL,
    review_notes TEXT,
    
    -- Audit Fields
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    -- Indexes
    INDEX idx_company_name (company_name),
    INDEX idx_contact_email (contact_email),
    INDEX idx_partnership_type (partnership_type),
    INDEX idx_status (status),
    INDEX idx_created_at (created_at),
    
    -- Foreign Keys
    FOREIGN KEY (reviewed_by) REFERENCES users(id)
);

-- ==================== CHAT SYSTEM ====================

CREATE TABLE chat_rooms (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    uuid CHAR(36) NOT NULL UNIQUE DEFAULT (UUID()),
    
    -- Room Details
    room_name VARCHAR(255),
    room_type ENUM('direct', 'group', 'support') DEFAULT 'direct',
    
    -- Participants
    participant_1_id BIGINT NOT NULL,
    participant_2_id BIGINT NULL,
    participants JSON, -- For group chats
    
    -- Room Settings
    is_active BOOLEAN DEFAULT TRUE,
    last_message_at DATETIME,
    
    -- Audit Fields
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    -- Indexes
    INDEX idx_participant_1 (participant_1_id),
    INDEX idx_participant_2 (participant_2_id),
    INDEX idx_room_type (room_type),
    INDEX idx_active (is_active),
    INDEX idx_last_message (last_message_at),
    
    -- Foreign Keys
    FOREIGN KEY (participant_1_id) REFERENCES users(id),
    FOREIGN KEY (participant_2_id) REFERENCES users(id)
);

CREATE TABLE chat_messages (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    uuid CHAR(36) NOT NULL UNIQUE DEFAULT (UUID()),
    
    -- Message Details
    room_id BIGINT NOT NULL,
    sender_id BIGINT NOT NULL,
    message_text TEXT NOT NULL,
    message_type ENUM('text', 'image', 'file', 'system') DEFAULT 'text',
    
    -- Attachments
    attachments JSON,
    
    -- Status
    is_read BOOLEAN DEFAULT FALSE,
    read_at DATETIME,
    is_edited BOOLEAN DEFAULT FALSE,
    edited_at DATETIME,
    
    -- Audit Fields
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    -- Indexes
    INDEX idx_room (room_id),
    INDEX idx_sender (sender_id),
    INDEX idx_created (created_at),
    INDEX idx_read (is_read),
    
    -- Foreign Keys
    FOREIGN KEY (room_id) REFERENCES chat_rooms(id) ON DELETE CASCADE,
    FOREIGN KEY (sender_id) REFERENCES users(id)
);

-- ==================== TRANSPORTATION SYSTEM ====================

CREATE TABLE drivers (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    uuid CHAR(36) NOT NULL UNIQUE DEFAULT (UUID()),
    
    -- Driver Information (references users table)
    user_id BIGINT NOT NULL UNIQUE,
    
    -- Driver Specific Details
    license_number VARCHAR(100) NOT NULL,
    license_expiry DATE NOT NULL,
    vehicle_registration VARCHAR(100),
    
    -- Status
    is_verified BOOLEAN DEFAULT FALSE,
    is_online BOOLEAN DEFAULT FALSE,
    is_available BOOLEAN DEFAULT TRUE,
    
    -- Performance Metrics
    total_trips INT DEFAULT 0,
    completed_trips INT DEFAULT 0,
    cancelled_trips INT DEFAULT 0,
    average_rating DECIMAL(3,2) DEFAULT 0,
    total_earnings DECIMAL(10,2) DEFAULT 0,
    
    -- Current Location
    current_latitude DECIMAL(10,8),
    current_longitude DECIMAL(11,8),
    last_location_update DATETIME,
    
    -- Audit Fields
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    -- Indexes
    INDEX idx_user (user_id),
    INDEX idx_verified (is_verified),
    INDEX idx_online (is_online),
    INDEX idx_available (is_available),
    INDEX idx_location (current_latitude, current_longitude),
    INDEX idx_rating (average_rating),
    
    -- Foreign Keys
    FOREIGN KEY (user_id) REFERENCES users(id)
);

CREATE TABLE transportation_vehicles (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    uuid CHAR(36) NOT NULL UNIQUE DEFAULT (UUID()),
    
    -- Vehicle Details
    driver_id BIGINT NOT NULL,
    vehicle_type ENUM('pickup_truck', 'van', 'box_truck', 'flatbed') NOT NULL,
    make VARCHAR(100) NOT NULL,
    model VARCHAR(100) NOT NULL,
    year YEAR NOT NULL,
    color VARCHAR(50),
    
    -- Registration
    license_plate VARCHAR(20) NOT NULL,
    registration_number VARCHAR(100),
    
    -- Specifications
    capacity_weight DECIMAL(8,2), -- in kg
    capacity_volume DECIMAL(8,2), -- in cubic meters
    
    -- Status
    is_active BOOLEAN DEFAULT TRUE,
    
    -- Audit Fields
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    -- Indexes
    INDEX idx_driver (driver_id),
    INDEX idx_vehicle_type (vehicle_type),
    INDEX idx_license_plate (license_plate),
    INDEX idx_active (is_active),
    
    -- Foreign Keys
    FOREIGN KEY (driver_id) REFERENCES drivers(id)
);

CREATE TABLE driver_bookings (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    uuid CHAR(36) NOT NULL UNIQUE DEFAULT (UUID()),
    
    -- Booking Details
    customer_id BIGINT NOT NULL,
    driver_id BIGINT NULL,
    vehicle_id BIGINT NULL,
    
    -- Service Details
    service_type ENUM('moving', 'delivery', 'transport') DEFAULT 'moving',
    vehicle_type ENUM('pickup_truck', 'van', 'box_truck', 'flatbed') NOT NULL,
    
    -- Locations
    pickup_address TEXT NOT NULL,
    pickup_latitude DECIMAL(10,8),
    pickup_longitude DECIMAL(11,8),
    dropoff_address TEXT NOT NULL,
    dropoff_latitude DECIMAL(10,8),
    dropoff_longitude DECIMAL(11,8),
    
    -- Timing
    scheduled_date DATE NOT NULL,
    scheduled_time TIME NOT NULL,
    estimated_duration INT, -- in minutes
    
    -- Pricing
    estimated_cost DECIMAL(10,2) NOT NULL,
    final_cost DECIMAL(10,2),
    distance_km DECIMAL(8,2),
    
    -- Status
    status ENUM('pending', 'confirmed', 'in_progress', 'completed', 'cancelled') DEFAULT 'pending',
    
    -- Additional Details
    special_instructions TEXT,
    items_description TEXT,
    
    -- Tracking
    started_at DATETIME,
    completed_at DATETIME,
    
    -- Rating & Feedback
    customer_rating INT,
    customer_feedback TEXT,
    driver_rating INT,
    driver_feedback TEXT,
    
    -- Audit Fields
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    -- Indexes
    INDEX idx_customer (customer_id),
    INDEX idx_driver (driver_id),
    INDEX idx_vehicle (vehicle_id),
    INDEX idx_status (status),
    INDEX idx_scheduled (scheduled_date, scheduled_time),
    INDEX idx_service_type (service_type),
    INDEX idx_vehicle_type (vehicle_type),
    
    -- Foreign Keys
    FOREIGN KEY (customer_id) REFERENCES users(id),
    FOREIGN KEY (driver_id) REFERENCES drivers(id),
    FOREIGN KEY (vehicle_id) REFERENCES transportation_vehicles(id)
);

CREATE TABLE driver_locations (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    
    -- Location Details
    driver_id BIGINT NOT NULL,
    booking_id BIGINT NULL,
    
    -- GPS Coordinates
    latitude DECIMAL(10,8) NOT NULL,
    longitude DECIMAL(11,8) NOT NULL,
    accuracy DECIMAL(8,2),
    
    -- Additional Info
    speed DECIMAL(5,2), -- km/h
    heading DECIMAL(5,2), -- degrees
    
    -- Timestamp
    recorded_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    
    -- Indexes
    INDEX idx_driver (driver_id),
    INDEX idx_booking (booking_id),
    INDEX idx_recorded (recorded_at),
    
    -- Foreign Keys
    FOREIGN KEY (driver_id) REFERENCES drivers(id),
    FOREIGN KEY (booking_id) REFERENCES driver_bookings(id)
);

-- ==================== SECURITY & AUDIT ====================

CREATE TABLE audit_logs (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    
    -- Audit Details
    user_id BIGINT,
    action VARCHAR(100) NOT NULL,
    entity_type ENUM('user', 'property', 'lease', 'payment', 'maintenance', 'document', 'review', 'vendor', 'partnership', 'chat', 'booking') NOT NULL,
    entity_id BIGINT NOT NULL,
    
    -- Changes
    old_values JSON,
    new_values JSON,
    
    -- Context
    ip_address VARCHAR(45),
    user_agent TEXT,
    description TEXT,
    affected_user_id BIGINT,
    
    -- Timestamp
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    
    -- Indexes
    INDEX idx_user (user_id),
    INDEX idx_entity (entity_type, entity_id),
    INDEX idx_action (action),
    INDEX idx_created (created_at),
    
    -- Foreign Keys
    FOREIGN KEY (user_id) REFERENCES users(id),
    FOREIGN KEY (affected_user_id) REFERENCES users(id)
);

CREATE TABLE system_settings (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    setting_key VARCHAR(255) NOT NULL UNIQUE,
    setting_value TEXT,
    setting_type VARCHAR(50) DEFAULT 'string',
    description TEXT,
    is_public BOOLEAN DEFAULT FALSE,
    updated_by BIGINT,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    INDEX idx_key (setting_key),
    FOREIGN KEY (updated_by) REFERENCES users(id)
);

-- ==================== INSERT DEFAULT DATA ====================

-- Insert User Roles (including driver)
INSERT INTO user_roles (role_code, role_name, description) VALUES
('super_admin', 'Super Administrator', 'Full platform access and system management'),
('system_admin', 'System Administrator', 'Enhanced system administrator with user management'),
('landlord', 'Property Owner/Landlord', 'Property owners and investors'),
('tenant', 'Tenant/Resident', 'Property residents and tenants'),
('property_manager', 'Property Manager', 'Professional property management'),
('property_agent', 'Property Agent', 'Real estate agents and brokers'),
('regional_manager', 'Regional Manager', 'Multi-location property management'),
('leasing_consultant', 'Leasing Consultant', 'Tenant acquisition specialists'),
('maintenance_supervisor', 'Maintenance Supervisor', 'Maintenance team leadership'),
('maintenance_staff', 'Maintenance Staff', 'Maintenance and repair technicians'),
('financial_officer', 'Financial Officer', 'Financial management and oversight'),
('financial_analyst', 'Financial Analyst', 'Financial analysis and reporting'),
('marketing_specialist', 'Marketing Specialist', 'Property marketing and advertising'),
('legal_advisor', 'Legal Advisor', 'Legal compliance and advice'),
('insurance_coordinator', 'Insurance Coordinator', 'Insurance policy management'),
('relocation_specialist', 'Relocation Specialist', 'Tenant relocation services'),
('community_manager', 'Community Manager', 'Apartment community management'),
('inspector', 'Inspector/Quality Assurance', 'Property inspections and quality control'),
('vendor', 'Vendor/Contractor', 'External service providers'),
('bnb_host', 'BNB Host', 'Short-term rental hosts'),
('corporate_housing_manager', 'Corporate Housing Manager', 'Corporate housing management'),
('student_housing_coordinator', 'Student Housing Coordinator', 'Student housing specialists'),
('luxury_property_specialist', 'Luxury Property Specialist', 'High-end property management'),
('data_analyst', 'Data Analyst', 'Business intelligence and analytics'),
('customer_support_agent', 'Customer Support Agent', 'User support and assistance'),
('driver', 'Transportation Driver', 'Moving and transportation services');

-- Insert Property Types
INSERT INTO property_types (type_code, type_name, description) VALUES
('apartment', 'Apartment', 'Multi-unit residential building'),
('house', 'Single-Family Home', 'Detached single-family residence'),
('condo', 'Condominium', 'Individually owned unit in a complex'),
('townhouse', 'Townhouse', 'Multi-floor attached home'),
('studio', 'Studio', 'Single-room apartment'),
('villa', 'Villa', 'Luxury detached residence'),
('multi_family', 'Multi-Family', 'Building with multiple residential units'),
('commercial', 'Commercial Space', 'Business and commercial properties'),
('other', 'Other', 'Other property types');

-- Insert Property Statuses
INSERT INTO property_statuses (status_code, status_name, description) VALUES
('draft', 'Draft', 'Property listing not published'),
('available', 'Available', 'Property available for rent'),
('occupied', 'Occupied', 'Property currently rented'),
('maintenance', 'Under Maintenance', 'Property undergoing maintenance'),
('unavailable', 'Unavailable', 'Property not available for rent');

-- Insert Maintenance Priorities
INSERT INTO maintenance_priorities (priority_code, priority_name, description, sla_hours) VALUES
('emergency', 'Emergency', 'Immediate attention required - safety hazards', 2),
('urgent', 'Urgent', 'Critical issues affecting livability', 24),
('high', 'High', 'Important but not critical issues', 72),
('normal', 'Normal', 'Routine maintenance requests', 168),
('low', 'Low', 'Cosmetic or minor issues', 336);

-- Insert Maintenance Statuses
INSERT INTO maintenance_statuses (status_code, status_name, description) VALUES
('submitted', 'Submitted', 'Request submitted by tenant'),
('assigned', 'Assigned', 'Assigned to maintenance staff'),
('in_progress', 'In Progress', 'Work in progress'),
('completed', 'Completed', 'Work completed successfully'),
('cancelled', 'Cancelled', 'Request cancelled'),
('on_hold', 'On Hold', 'Work temporarily on hold');

-- Insert Lease Statuses
INSERT INTO lease_statuses (status_code, status_name, description) VALUES
('draft', 'Draft', 'Lease agreement in draft stage'),
('pending', 'Pending', 'Awaiting signature or approval'),
('active', 'Active', 'Lease currently in effect'),
('expired', 'Expired', 'Lease term completed'),
('terminated', 'Terminated', 'Lease ended prematurely'),
('renewed', 'Renewed', 'Lease extended for new term');

-- Insert Payment Statuses
INSERT INTO payment_statuses (status_code, status_name, description) VALUES
('pending', 'Pending', 'Payment initiated but not completed'),
('paid', 'Paid', 'Payment successfully completed'),
('failed', 'Failed', 'Payment failed or declined'),
('overdue', 'Overdue', 'Payment past due date'),
('partial', 'Partial', 'Partial payment received'),
('refunded', 'Refunded', 'Payment refunded to tenant');

-- Insert System Settings
INSERT INTO system_settings (setting_key, setting_value, setting_type, description, is_public) VALUES
('company_name', 'StaySpot', 'string', 'Platform company name', TRUE),
('company_location', 'Nakuru, Kenya', 'string', 'Company headquarters location', TRUE),
('company_phone', '+254748041595', 'string', 'Company contact phone', TRUE),
('platform_version', '3.0.0', 'string', 'Current platform version', TRUE),
('service_areas', 'Nairobi, Mombasa, Kisumu, Nakuru, Eldoret, Thika, Machakos, Meru, Nyeri, Kakamega, Kitale, Garissa, Malindi, Lamu, Naivasha, Nanyuki', 'string', 'Cities and towns where StaySpot operates', TRUE),
('default_currency', 'KES', 'string', 'Default currency for payments', TRUE),
('rent_due_day', '1', 'number', 'Default day of month rent is due', FALSE),
('late_fee_percentage', '5.0', 'number', 'Default late fee percentage', FALSE),
('grace_period_days', '5', 'number', 'Default grace period for late payments', FALSE),
('maintenance_emergency_hours', '2', 'number', 'SLA for emergency maintenance', FALSE),
('maintenance_urgent_hours', '24', 'number', 'SLA for urgent maintenance', FALSE),
('auto_archive_days', '90', 'number', 'Days before archiving old records', FALSE),
('chat_message_retention_days', '365', 'number', 'Days to retain chat messages', FALSE),
('driver_location_update_interval', '30', 'number', 'Seconds between location updates', FALSE);

-- ==================== CREATE SAMPLE DATA ====================

-- Note: Passwords are hashed version of 'password123'
INSERT INTO users (email, password_hash, first_name, last_name, phone, role_code, is_email_verified, is_active, company_name, address_country, created_at) VALUES
-- Administration
('admin@stayspot.co.ke', '$2a$12$LQv3c1yqBWVHxkd0L6kZrOeG5c4jU8VzLk8VZcFgKfM8nE9zL8bK', 'Super', 'Admin', '+254748041595', 'super_admin', TRUE, TRUE, 'StaySpot Kenya', 'KE', NOW()),

-- Property Owners
('landlord@example.co.ke', '$2a$12$LQv3c1yqBWVHxkd0L6kZrOeG5c4jU8VzLk8VZcFgKfM8nE9zL8bK', 'John', 'Kamau', '+254712345678', 'landlord', TRUE, TRUE, 'Kamau Properties', 'KE', NOW()),

-- Property Manager
('manager@stayspot.co.ke', '$2a$12$LQv3c1yqBWVHxkd0L6kZrOeG5c4jU8VzLk8VZcFgKfM8nE9zL8bK', 'Grace', 'Wanjiku', '+254723456789', 'property_manager', TRUE, TRUE, 'StaySpot Management', 'KE', NOW()),

-- Tenants
('tenant1@example.co.ke', '$2a$12$LQv3c1yqBWVHxkd0L6kZrOeG5c4jU8VzLk8VZcFgKfM8nE9zL8bK', 'Mary', 'Njeri', '+254734567890', 'tenant', TRUE, TRUE, NULL, 'KE', NOW()),
('tenant2@example.co.ke', '$2a$12$LQv3c1yqBWVHxkd0L6kZrOeG5c4jU8VzLk8VZcFgKfM8nE9zL8bK', 'Peter', 'Mwangi', '+254745678901', 'tenant', TRUE, TRUE, NULL, 'KE', NOW()),

-- Driver
('driver@stayspot.co.ke', '$2a$12$LQv3c1yqBWVHxkd0L6kZrOeG5c4jU8VzLk8VZcFgKfM8nE9zL8bK', 'James', 'Kiprotich', '+254756789012', 'driver', TRUE, TRUE, 'StaySpot Transport', 'KE', NOW());

-- Create sample properties across major Kenyan cities
INSERT INTO properties (title, description, property_type_code, status_code, address_street, address_city, address_state, address_zip_code, address_country, bedrooms, bathrooms, area_sqft, monthly_rent, security_deposit, amenities, owner_id, is_published, created_by, created_at) VALUES
('Modern Apartment in Nakuru CBD', 'Beautiful 2-bedroom apartment in the heart of Nakuru with modern amenities and city views.', 'apartment', 'available', 'Kenyatta Avenue', 'Nakuru', 'Nakuru County', '20100', 'KE', 2, 2, 1000, 25000.00, 25000.00, '["wifi", "parking", "security", "water"]', 2, TRUE, 2, NOW()),
('Luxury Apartment in Westlands', 'Premium 3-bedroom apartment in Westlands, Nairobi with gym and swimming pool.', 'apartment', 'available', 'Westlands Road', 'Nairobi', 'Nairobi County', '00100', 'KE', 3, 2, 1400, 85000.00, 85000.00, '["wifi", "parking", "gym", "pool", "security"]', 2, TRUE, 2, NOW()),
('Beachfront Villa in Mombasa', 'Stunning 4-bedroom villa near Diani Beach with ocean views and private garden.', 'villa', 'available', 'Diani Beach Road', 'Mombasa', 'Mombasa County', '80100', 'KE', 4, 3, 2200, 120000.00, 120000.00, '["wifi", "parking", "garden", "beach_access", "security"]', 2, TRUE, 2, NOW()),
('Family House in Milimani', 'Spacious 3-bedroom house in quiet Milimani estate with garden and parking.', 'house', 'available', 'Milimani Estate', 'Nakuru', 'Nakuru County', '20100', 'KE', 3, 2, 1500, 35000.00, 35000.00, '["parking", "garden", "security", "water"]', 2, TRUE, 2, NOW()),
('Student Apartment in Eldoret', 'Affordable 1-bedroom apartment near Moi University, perfect for students.', 'apartment', 'available', 'University Road', 'Eldoret', 'Uasin Gishu County', '30100', 'KE', 1, 1, 600, 15000.00, 15000.00, '["wifi", "security", "water"]', 2, TRUE, 2, NOW());

-- Create sample driver
INSERT INTO drivers (user_id, license_number, license_expiry, vehicle_registration, is_verified, is_available, current_latitude, current_longitude, last_location_update) VALUES
(6, 'DL123456789', '2025-12-31', 'KCA123D', TRUE, TRUE, -0.3031, 36.0800, NOW());

-- Create sample vehicle
INSERT INTO transportation_vehicles (driver_id, vehicle_type, make, model, year, color, license_plate, capacity_weight, capacity_volume, is_active) VALUES
(1, 'pickup_truck', 'Toyota', 'Hilux', 2020, 'White', 'KCA123D', 1000.00, 5.00, TRUE);

-- Create sample chat room
INSERT INTO chat_rooms (participant_1_id, participant_2_id, room_type, is_active, last_message_at) VALUES
(4, 2, 'direct', TRUE, NOW());

-- Create sample chat message
INSERT INTO chat_messages (room_id, sender_id, message_text, message_type, is_read) VALUES
(1, 4, 'Hello, I am interested in the apartment listing.', 'text', FALSE);

-- ==================== CREATE VIEWS ====================

CREATE VIEW property_overview AS
SELECT 
    p.id,
    p.uuid,
    p.title,
    p.property_type_code,
    pt.type_name as property_type,
    p.status_code,
    ps.status_name as property_status,
    p.address_street,
    p.address_city,
    p.address_state,
    p.monthly_rent,
    p.bedrooms,
    p.bathrooms,
    p.area_sqft,
    p.available_from,
    CONCAT(u_owner.first_name, ' ', u_owner.last_name) as owner_name,
    CONCAT(u_manager.first_name, ' ', u_manager.last_name) as manager_name,
    (SELECT COUNT(*) FROM leases l WHERE l.property_id = p.id AND l.status_code = 'active' AND l.deleted_at IS NULL) as active_leases,
    (SELECT COUNT(*) FROM maintenance_requests mr WHERE mr.property_id = p.id AND mr.status_code != 'completed') as open_maintenance_requests
FROM properties p
LEFT JOIN property_types pt ON p.property_type_code = pt.type_code
LEFT JOIN property_statuses ps ON p.status_code = ps.status_code
LEFT JOIN users u_owner ON p.owner_id = u_owner.id
LEFT JOIN users u_manager ON p.property_manager_id = u_manager.id
WHERE p.deleted_at IS NULL;

CREATE VIEW driver_overview AS
SELECT 
    d.id,
    d.uuid,
    CONCAT(u.first_name, ' ', u.last_name) as driver_name,
    u.email,
    u.phone,
    d.license_number,
    d.is_verified,
    d.is_online,
    d.is_available,
    d.total_trips,
    d.completed_trips,
    d.average_rating,
    d.total_earnings,
    d.current_latitude,
    d.current_longitude,
    d.last_location_update,
    v.vehicle_type,
    v.make,
    v.model,
    v.license_plate
FROM drivers d
JOIN users u ON d.user_id = u.id
LEFT JOIN transportation_vehicles v ON d.id = v.driver_id AND v.is_active = TRUE;

-- ==================== FINAL SETUP ====================

-- Display completion message
SELECT 'StaySpot Enhanced Database Schema Created Successfully!' as message;

-- Show table counts
SELECT 
    COUNT(*) as total_tables_created 
FROM information_schema.tables 
WHERE table_schema = 'stayspotco_stayspot' 
AND table_type = 'BASE TABLE';

-- Show sample data counts
SELECT 
    'Users' as table_name, COUNT(*) as count FROM users
UNION ALL
    SELECT 'Properties', COUNT(*) FROM properties
UNION ALL
    SELECT 'Drivers', COUNT(*) FROM drivers
UNION ALL
    SELECT 'Chat Rooms', COUNT(*) FROM chat_rooms
UNION ALL
    SELECT 'Chat Messages', COUNT(*) FROM chat_messages
UNION ALL
    SELECT 'Transportation Vehicles', COUNT(*) FROM transportation_vehicles;