-- StaySpot Additional Features
-- Emergency contacts, property analytics, and mobile app support

USE stayspotco_stayspot;

-- ==================== EMERGENCY CONTACTS SYSTEM ====================

CREATE TABLE IF NOT EXISTS emergency_contacts (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    property_id BIGINT NOT NULL,
    contact_type ENUM('police', 'hospital', 'fire', 'security', 'utility', 'maintenance') NOT NULL,
    contact_name VARCHAR(255) NOT NULL,
    phone_number VARCHAR(20) NOT NULL,
    address TEXT,
    is_24_7 BOOLEAN DEFAULT FALSE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    
    INDEX idx_property (property_id),
    INDEX idx_contact_type (contact_type),
    FOREIGN KEY (property_id) REFERENCES properties(id)
);

-- ==================== PROPERTY ANALYTICS ====================

CREATE TABLE IF NOT EXISTS property_analytics (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    property_id BIGINT NOT NULL,
    metric_date DATE NOT NULL,
    views_count INT DEFAULT 0,
    inquiries_count INT DEFAULT 0,
    applications_count INT DEFAULT 0,
    occupancy_rate DECIMAL(5,2) DEFAULT 0,
    revenue DECIMAL(10,2) DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    
    UNIQUE KEY unique_property_date (property_id, metric_date),
    INDEX idx_property (property_id),
    INDEX idx_date (metric_date),
    FOREIGN KEY (property_id) REFERENCES properties(id)
);

-- ==================== MOBILE APP TOKENS ====================

CREATE TABLE IF NOT EXISTS mobile_app_tokens (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    user_id BIGINT NOT NULL,
    device_token VARCHAR(255) NOT NULL,
    device_type ENUM('ios', 'android') NOT NULL,
    app_version VARCHAR(20),
    is_active BOOLEAN DEFAULT TRUE,
    last_used DATETIME,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    
    INDEX idx_user (user_id),
    INDEX idx_token (device_token),
    INDEX idx_active (is_active),
    FOREIGN KEY (user_id) REFERENCES users(id)
);

-- ==================== PROPERTY FAVORITES ====================

CREATE TABLE IF NOT EXISTS property_favorites (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    user_id BIGINT NOT NULL,
    property_id BIGINT NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    
    UNIQUE KEY unique_user_property (user_id, property_id),
    INDEX idx_user (user_id),
    INDEX idx_property (property_id),
    FOREIGN KEY (user_id) REFERENCES users(id),
    FOREIGN KEY (property_id) REFERENCES properties(id)
);

-- ==================== RENT REMINDERS ====================

CREATE TABLE IF NOT EXISTS rent_reminders (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    lease_id BIGINT NOT NULL,
    reminder_date DATE NOT NULL,
    reminder_type ENUM('7_days', '3_days', '1_day', 'due_date', 'overdue') NOT NULL,
    is_sent BOOLEAN DEFAULT FALSE,
    sent_at DATETIME,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    
    INDEX idx_lease (lease_id),
    INDEX idx_reminder_date (reminder_date),
    INDEX idx_sent (is_sent),
    FOREIGN KEY (lease_id) REFERENCES leases(id)
);

-- ==================== INSERT SAMPLE DATA ====================

-- Emergency contacts for sample properties
INSERT INTO emergency_contacts (property_id, contact_type, contact_name, phone_number, is_24_7) VALUES
(1, 'police', 'Nakuru Police Station', '+254202345678', TRUE),
(1, 'hospital', 'Nakuru Level 5 Hospital', '+254202345679', TRUE),
(1, 'security', 'Property Security', '+254712345678', TRUE),
(2, 'police', 'Nairobi Central Police', '+254202123456', TRUE),
(2, 'hospital', 'Kenyatta National Hospital', '+254202726300', TRUE);

-- Sample analytics data
INSERT INTO property_analytics (property_id, metric_date, views_count, inquiries_count, occupancy_rate, revenue) VALUES
(1, CURDATE() - INTERVAL 1 DAY, 45, 8, 100.00, 25000.00),
(1, CURDATE() - INTERVAL 2 DAY, 38, 6, 100.00, 25000.00),
(2, CURDATE() - INTERVAL 1 DAY, 62, 12, 100.00, 85000.00);

-- ==================== UPDATE SYSTEM SETTINGS ====================

INSERT IGNORE INTO system_settings (setting_key, setting_value, setting_type, description, is_public) VALUES
('emergency_hotline', '999', 'string', 'National emergency hotline', TRUE),
('support_hours', '24/7', 'string', 'Customer support availability', TRUE),
('mobile_app_version', '3.0.0', 'string', 'Current mobile app version', TRUE),
('push_notifications_enabled', 'true', 'boolean', 'Enable push notifications', FALSE),
('analytics_retention_days', '365', 'number', 'Days to retain analytics data', FALSE);

SELECT 'Additional features added successfully!' as message;