const { Parser } = require('json2csv');

const reportGenerator = {
  // Generate payment report in CSV format
  generatePaymentReport: (payments, format = 'csv') => {
    try {
      const csvData = payments.map(payment => ({
        'Payment ID': payment.id,
        'Date': payment.paidDate ? new Date(payment.paidDate).toISOString().split('T')[0] : '',
        'Property': payment.property?.title || '',
        'Address': payment.property?.address || '',
        'Tenant': `${payment.tenant?.firstName || ''} ${payment.tenant?.lastName || ''}`.trim(),
        'Amount': payment.amount,
        'Status': payment.status,
        'Payment Method': payment.paymentMethod || '',
        'Due Date': payment.dueDate ? new Date(payment.dueDate).toISOString().split('T')[0] : '',
        'Description': payment.description || ''
      }));

      if (format === 'csv') {
        const parser = new Parser();
        return parser.parse(csvData);
      }

      return csvData;
    } catch (error) {
      console.error('Generate payment report error:', error);
      throw new Error('Failed to generate payment report');
    }
  },

  // Generate maintenance report
  generateMaintenanceReport: (maintenanceRequests) => {
    return maintenanceRequests.map(request => ({
      'Request ID': request.id,
      'Title': request.title,
      'Property': request.property?.title || '',
      'Address': request.property?.address || '',
      'Tenant': `${request.tenant?.firstName || ''} ${request.tenant?.lastName || ''}`.trim(),
      'Category': request.category,
      'Priority': request.priority,
      'Status': request.status,
      'Created Date': new Date(request.createdAt).toISOString().split('T')[0],
      'Completed Date': request.completedDate ? new Date(request.completedDate).toISOString().split('T')[0] : '',
      'Cost': request.cost || '0',
      'Assigned To': request.assignedStaff ? 
        `${request.assignedStaff.firstName} ${request.assignedStaff.lastName}` : 'Unassigned'
    }));
  },

  // Generate lease report
  generateLeaseReport: (leases) => {
    return leases.map(lease => ({
      'Lease ID': lease.id,
      'Property': lease.property?.title || '',
      'Address': lease.property?.address || '',
      'Tenant': `${lease.tenant?.firstName || ''} ${lease.tenant?.lastName || ''}`.trim(),
      'Start Date': new Date(lease.startDate).toISOString().split('T')[0],
      'End Date': new Date(lease.endDate).toISOString().split('T')[0],
      'Monthly Rent': lease.monthlyRent,
      'Security Deposit': lease.securityDeposit,
      'Status': lease.status,
      'Days Remaining': lease.daysRemaining ? lease.daysRemaining() : 'N/A'
    }));
  },

  // Generate financial summary
  generateFinancialSummary: (payments, startDate, endDate) => {
    const filteredPayments = payments.filter(payment => {
      const paymentDate = payment.paidDate ? new Date(payment.paidDate) : new Date(payment.dueDate);
      return (!startDate || paymentDate >= new Date(startDate)) && 
             (!endDate || paymentDate <= new Date(endDate));
    });

    const totalRevenue = filteredPayments
      .filter(p => p.status === 'paid')
      .reduce((sum, p) => sum + parseFloat(p.amount), 0);

    const pendingPayments = filteredPayments
      .filter(p => p.status === 'pending')
      .reduce((sum, p) => sum + parseFloat(p.amount), 0);

    const paymentMethods = filteredPayments.reduce((acc, payment) => {
      const method = payment.paymentMethod || 'unknown';
      acc[method] = (acc[method] || 0) + 1;
      return acc;
    }, {});

    return {
      period: {
        startDate: startDate || 'Beginning',
        endDate: endDate || 'Present'
      },
      totalRevenue,
      pendingPayments,
      totalTransactions: filteredPayments.length,
      paidTransactions: filteredPayments.filter(p => p.status === 'paid').length,
      paymentMethods,
      generatedAt: new Date().toISOString()
    };
  }
};

module.exports = reportGenerator;