const axios = require('axios');
const NodeGeocoder = require('node-geocoder');

/**
 * Geocoding Utility
 * Handles address geocoding and reverse geocoding
 */

class Geocoder {
  constructor() {
    this.provider = process.env.GEOCODER_PROVIDER || 'google';
    this.apiKey = process.env.GOOGLE_MAPS_API_KEY;
    
    this.geocoder = NodeGeocoder({
      provider: this.provider,
      apiKey: this.apiKey,
      formatter: null
    });
  }

  // Geocode an address to get coordinates
  async geocodeAddress(address) {
    try {
      const results = await this.geocoder.geocode(address);
      
      if (results && results.length > 0) {
        const result = results[0];
        
        return {
          success: true,
          latitude: result.latitude,
          longitude: result.longitude,
          formattedAddress: result.formattedAddress,
          street: result.streetName,
          streetNumber: result.streetNumber,
          city: result.city,
          state: result.state,
          stateCode: result.stateCode,
          zipcode: result.zipcode,
          country: result.country,
          countryCode: result.countryCode,
          provider: this.provider
        };
      }
      
      return {
        success: false,
        error: 'No results found for the given address'
      };
    } catch (error) {
      console.error('Geocoding error:', error);
      return {
        success: false,
        error: error.message
      };
    }
  }

  // Reverse geocode coordinates to get address
  async reverseGeocode(latitude, longitude) {
    try {
      const results = await this.geocoder.reverse({ lat: latitude, lon: longitude });
      
      if (results && results.length > 0) {
        const result = results[0];
        
        return {
          success: true,
          latitude: result.latitude,
          longitude: result.longitude,
          formattedAddress: result.formattedAddress,
          street: result.streetName,
          streetNumber: result.streetNumber,
          city: result.city,
          state: result.state,
          stateCode: result.stateCode,
          zipcode: result.zipcode,
          country: result.country,
          countryCode: result.countryCode,
          provider: this.provider
        };
      }
      
      return {
        success: false,
        error: 'No results found for the given coordinates'
      };
    } catch (error) {
      console.error('Reverse geocoding error:', error);
      return {
        success: false,
        error: error.message
      };
    }
  }

  // Batch geocode multiple addresses
  async batchGeocode(addresses) {
    try {
      const results = [];
      
      for (const address of addresses) {
        const result = await this.geocodeAddress(address);
        results.push({
          address,
          ...result
        });
        
        // Add delay to respect rate limits
        await new Promise(resolve => setTimeout(resolve, 100));
      }
      
      return {
        success: true,
        results
      };
    } catch (error) {
      console.error('Batch geocoding error:', error);
      return {
        success: false,
        error: error.message
      };
    }
  }

  // Calculate distance between two coordinates (Haversine formula)
  calculateDistance(lat1, lon1, lat2, lon2, unit = 'miles') {
    const R = unit === 'miles' ? 3959 : 6371; // Earth radius in miles or kilometers
    const dLat = this.toRad(lat2 - lat1);
    const dLon = this.toRad(lon2 - lon1);
    
    const a = 
      Math.sin(dLat / 2) * Math.sin(dLat / 2) +
      Math.cos(this.toRad(lat1)) * Math.cos(this.toRad(lat2)) * 
      Math.sin(dLon / 2) * Math.sin(dLon / 2);
    
    const c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1 - a));
    const distance = R * c;
    
    return parseFloat(distance.toFixed(2));
  }

  // Convert degrees to radians
  toRad(degrees) {
    return degrees * (Math.PI / 180);
  }

  // Find properties within radius
  async findPropertiesInRadius(centerLat, centerLon, radius, properties) {
    const results = [];
    
    for (const property of properties) {
      if (property.latitude && property.longitude) {
        const distance = this.calculateDistance(
          centerLat, centerLon, 
          property.latitude, property.longitude,
          'miles'
        );
        
        if (distance <= radius) {
          results.push({
            ...property.toJSON ? property.toJSON() : property,
            distance: parseFloat(distance.toFixed(2))
          });
        }
      }
    }
    
    // Sort by distance
    results.sort((a, b) => a.distance - b.distance);
    
    return results;
  }

  // Validate coordinates
  validateCoordinates(latitude, longitude) {
    const lat = parseFloat(latitude);
    const lng = parseFloat(longitude);
    
    if (isNaN(lat) || isNaN(lng)) {
      return false;
    }
    
    return lat >= -90 && lat <= 90 && lng >= -180 && lng <= 180;
  }

  // Get address components from formatted address
  parseAddressComponents(formattedAddress) {
    // Basic parsing - in production, use a more robust solution
    const components = {
      street: '',
      city: '',
      state: '',
      zipcode: '',
      country: ''
    };
    
    try {
      const parts = formattedAddress.split(',');
      
      if (parts.length >= 1) components.street = parts[0].trim();
      if (parts.length >= 2) components.city = parts[1].trim();
      if (parts.length >= 3) {
        const stateZip = parts[2].trim().split(' ');
        if (stateZip.length >= 2) {
          components.state = stateZip[0];
          components.zipcode = stateZip[1];
        }
      }
      if (parts.length >= 4) components.country = parts[3].trim();
      
      return components;
    } catch (error) {
      console.error('Address parsing error:', error);
      return components;
    }
  }

  // Get timezone for coordinates
  async getTimezone(latitude, longitude) {
    try {
      if (!this.apiKey) {
        throw new Error('Google Maps API key is required for timezone lookup');
      }
      
      const response = await axios.get(
        `https://maps.googleapis.com/maps/api/timezone/json`,
        {
          params: {
            location: `${latitude},${longitude}`,
            timestamp: Math.floor(Date.now() / 1000),
            key: this.apiKey
          }
        }
      );
      
      if (response.data.status === 'OK') {
        return {
          success: true,
          timeZoneId: response.data.timeZoneId,
          timeZoneName: response.data.timeZoneName,
          rawOffset: response.data.rawOffset,
          dstOffset: response.data.dstOffset
        };
      }
      
      return {
        success: false,
        error: response.data.errorMessage || 'Timezone lookup failed'
      };
    } catch (error) {
      console.error('Timezone lookup error:', error);
      return {
        success: false,
        error: error.message
      };
    }
  }

  // Get elevation for coordinates
  async getElevation(latitude, longitude) {
    try {
      if (!this.apiKey) {
        throw new Error('Google Maps API key is required for elevation lookup');
      }
      
      const response = await axios.get(
        `https://maps.googleapis.com/maps/api/elevation/json`,
        {
          params: {
            locations: `${latitude},${longitude}`,
            key: this.apiKey
          }
        }
      );
      
      if (response.data.status === 'OK' && response.data.results.length > 0) {
        return {
          success: true,
          elevation: response.data.results[0].elevation,
          resolution: response.data.results[0].resolution
        };
      }
      
      return {
        success: false,
        error: response.data.errorMessage || 'Elevation lookup failed'
      };
    } catch (error) {
      console.error('Elevation lookup error:', error);
      return {
        success: false,
        error: error.message
      };
    }
  }

  // Get driving distance and duration between two points
  async getDrivingDistance(origin, destination) {
    try {
      if (!this.apiKey) {
        throw new Error('Google Maps API key is required for distance matrix');
      }
      
      const response = await axios.get(
        `https://maps.googleapis.com/maps/api/distancematrix/json`,
        {
          params: {
            origins: origin,
            destinations: destination,
            key: this.apiKey,
            units: 'imperial'
          }
        }
      );
      
      if (response.data.status === 'OK' && response.data.rows.length > 0) {
        const element = response.data.rows[0].elements[0];
        
        if (element.status === 'OK') {
          return {
            success: true,
            distance: {
              text: element.distance.text,
              value: element.distance.value // in meters
            },
            duration: {
              text: element.duration.text,
              value: element.duration.value // in seconds
            }
          };
        }
      }
      
      return {
        success: false,
        error: response.data.error_message || 'Distance calculation failed'
      };
    } catch (error) {
      console.error('Distance calculation error:', error);
      return {
        success: false,
        error: error.message
      };
    }
  }
}

module.exports = new Geocoder();