/**
 * CSV Export Utility
 * Simplified version for development
 */

class CSVExporter {
    // Convert JSON data to CSV string
    static jsonToCSV(data, fields = null) {
      try {
        if (!data || data.length === 0) {
          return '';
        }
  
        const actualFields = fields || Object.keys(data[0]);
        
        // Create header row
        let csv = actualFields.join(',') + '\n';
        
        // Create data rows
        data.forEach(item => {
          const row = actualFields.map(field => {
            let value = item[field];
            
            // Handle nested fields
            if (field.includes('.')) {
              const parts = field.split('.');
              value = item;
              for (const part of parts) {
                value = value?.[part];
              }
            }
            
            // Escape commas and quotes
            if (typeof value === 'string' && (value.includes(',') || value.includes('"'))) {
              value = `"${value.replace(/"/g, '""')}"`;
            }
            
            return value || '';
          });
          csv += row.join(',') + '\n';
        });
        
        return csv;
      } catch (error) {
        throw new Error(`CSV conversion failed: ${error.message}`);
      }
    }
  
    // Generate CSV for payments report
    static generatePaymentsCSV(payments) {
      const fields = [
        'id',
        'amount',
        'dueDate',
        'paidDate',
        'status',
        'paymentMethod',
        'tenant.firstName',
        'tenant.lastName',
        'property.title'
      ];
  
      return this.jsonToCSV(payments, fields);
    }
  
    // Generate CSV for properties report
    static generatePropertiesCSV(properties) {
      const fields = [
        'id',
        'title',
        'type',
        'price',
        'bedrooms',
        'bathrooms',
        'address',
        'city',
        'state',
        'status'
      ];
  
      return this.jsonToCSV(properties, fields);
    }
  
    // Generate CSV for leases report
    static generateLeasesCSV(leases) {
      const fields = [
        'id',
        'startDate',
        'endDate',
        'monthlyRent',
        'status',
        'property.title',
        'tenant.firstName',
        'tenant.lastName'
      ];
  
      return this.jsonToCSV(leases, fields);
    }
  
    // Generate CSV for maintenance report
    static generateMaintenanceCSV(maintenanceRequests) {
      const fields = [
        'id',
        'title',
        'priority',
        'status',
        'category',
        'createdAt',
        'property.title'
      ];
  
      return this.jsonToCSV(maintenanceRequests, fields);
    }
  }
  
  module.exports = CSVExporter;