const axios = require('axios');
const ical = require('ical');
const { GoogleAuth } = require('google-auth-library');

/**
 * Calendar Sync Utility
 * Handles synchronization with Google Calendar, Outlook, and other calendar services
 */

class CalendarSync {
  constructor() {
    this.googleClient = null;
    this.microsoftClient = null;
    
    this.initializeGoogleClient();
  }

  // Initialize Google Calendar client
  initializeGoogleClient() {
    try {
      if (process.env.GOOGLE_CALENDAR_CLIENT_EMAIL && process.env.GOOGLE_CALENDAR_PRIVATE_KEY) {
        this.googleClient = new GoogleAuth({
          credentials: {
            client_email: process.env.GOOGLE_CALENDAR_CLIENT_EMAIL,
            private_key: process.env.GOOGLE_CALENDAR_PRIVATE_KEY.replace(/\\n/g, '\n')
          },
          scopes: ['https://www.googleapis.com/auth/calendar']
        });
      }
    } catch (error) {
      console.error('Failed to initialize Google Calendar client:', error);
    }
  }

  // Sync lease dates to Google Calendar
  async syncToGoogleCalendar(lease, calendarId = 'primary') {
    try {
      if (!this.googleClient) {
        throw new Error('Google Calendar client not configured');
      }

      const authClient = await this.googleClient.getClient();
      
      const event = {
        summary: `Lease: ${lease.property.title}`,
        description: `Lease agreement for ${lease.property.address}. Tenant: ${lease.tenant.firstName} ${lease.tenant.lastName}. Monthly rent: $${lease.monthlyRent}`,
        start: {
          dateTime: new Date(lease.startDate).toISOString(),
          timeZone: 'UTC'
        },
        end: {
          dateTime: new Date(lease.endDate).toISOString(),
          timeZone: 'UTC'
        },
        reminders: {
          useDefault: false,
          overrides: [
            { method: 'email', minutes: 24 * 60 }, // 1 day before
            { method: 'popup', minutes: 60 } // 1 hour before
          ]
        },
        extendedProperties: {
          private: {
            leaseId: lease.id.toString(),
            propertyId: lease.propertyId.toString(),
            tenantId: lease.tenantId.toString()
          }
        }
      };

      const response = await axios.post(
        `https://www.googleapis.com/calendar/v3/calendars/${calendarId}/events`,
        event,
        {
          headers: {
            Authorization: `Bearer ${authClient.credentials.access_token}`,
            'Content-Type': 'application/json'
          }
        }
      );

      return {
        success: true,
        eventId: response.data.id,
        htmlLink: response.data.htmlLink,
        message: 'Lease synced to Google Calendar successfully'
      };
    } catch (error) {
      console.error('Google Calendar sync error:', error);
      return {
        success: false,
        error: error.response?.data?.error?.message || error.message
      };
    }
  }

  // Sync maintenance schedule to calendar
  async syncMaintenanceToCalendar(maintenance, calendarType = 'google', calendarId = 'primary') {
    try {
      const event = {
        summary: `Maintenance: ${maintenance.title}`,
        description: `Maintenance request for ${maintenance.property.address}. Priority: ${maintenance.priority}. Category: ${maintenance.category}`,
        start: {
          dateTime: maintenance.scheduledDate ? new Date(maintenance.scheduledDate).toISOString() : new Date().toISOString(),
          timeZone: 'UTC'
        },
        end: {
          dateTime: maintenance.scheduledDate ? 
            new Date(new Date(maintenance.scheduledDate).getTime() + 2 * 60 * 60 * 1000).toISOString() : // 2 hours duration
            new Date(new Date().getTime() + 2 * 60 * 60 * 1000).toISOString(),
          timeZone: 'UTC'
        },
        extendedProperties: {
          private: {
            maintenanceId: maintenance.id.toString(),
            propertyId: maintenance.propertyId.toString(),
            priority: maintenance.priority
          }
        }
      };

      if (calendarType === 'google') {
        return await this.syncToGoogleCalendarEvent(event, calendarId);
      } else if (calendarType === 'outlook') {
        return await this.syncToOutlookCalendar(event);
      } else {
        throw new Error(`Unsupported calendar type: ${calendarType}`);
      }
    } catch (error) {
      console.error('Maintenance calendar sync error:', error);
      return {
        success: false,
        error: error.message
      };
    }
  }

  // Sync payment due dates to calendar
  async syncPaymentToCalendar(payment, calendarType = 'google', calendarId = 'primary') {
    try {
      const event = {
        summary: `Rent Due: $${payment.amount}`,
        description: `Rent payment due for ${payment.property.address}. Lease: ${payment.leaseId}`,
        start: {
          dateTime: new Date(payment.dueDate).toISOString(),
          timeZone: 'UTC'
        },
        end: {
          dateTime: new Date(new Date(payment.dueDate).getTime() + 24 * 60 * 60 * 1000).toISOString(), // All day event
          timeZone: 'UTC'
        },
        extendedProperties: {
          private: {
            paymentId: payment.id.toString(),
            leaseId: payment.leaseId.toString(),
            amount: payment.amount.toString()
          }
        }
      };

      if (calendarType === 'google') {
        return await this.syncToGoogleCalendarEvent(event, calendarId);
      } else if (calendarType === 'outlook') {
        return await this.syncToOutlookCalendar(event);
      } else {
        throw new Error(`Unsupported calendar type: ${calendarType}`);
      }
    } catch (error) {
      console.error('Payment calendar sync error:', error);
      return {
        success: false,
        error: error.message
      };
    }
  }

  // Generic method to sync event to Google Calendar
  async syncToGoogleCalendarEvent(event, calendarId = 'primary') {
    try {
      if (!this.googleClient) {
        throw new Error('Google Calendar client not configured');
      }

      const authClient = await this.googleClient.getClient();
      
      const response = await axios.post(
        `https://www.googleapis.com/calendar/v3/calendars/${calendarId}/events`,
        event,
        {
          headers: {
            Authorization: `Bearer ${authClient.credentials.access_token}`,
            'Content-Type': 'application/json'
          }
        }
      );

      return {
        success: true,
        eventId: response.data.id,
        htmlLink: response.data.htmlLink,
        message: 'Event synced to Google Calendar successfully'
      };
    } catch (error) {
      console.error('Google Calendar event sync error:', error);
      return {
        success: false,
        error: error.response?.data?.error?.message || error.message
      };
    }
  }

  // Sync to Outlook Calendar (placeholder implementation)
  async syncToOutlookCalendar(event) {
    // Implementation for Outlook Calendar sync would go here
    // This requires Microsoft Graph API integration
    
    console.log('Outlook Calendar sync not implemented yet');
    return {
      success: false,
      error: 'Outlook Calendar sync not implemented'
    };
  }

  // Generate iCal file for lease
  generateLeaseICal(lease) {
    try {
      const icalEvent = {
        start: new Date(lease.startDate),
        end: new Date(lease.endDate),
        summary: `Lease: ${lease.property.title}`,
        description: `Lease agreement for ${lease.property.address}. Monthly rent: $${lease.monthlyRent}`,
        location: lease.property.address,
        url: `${process.env.FRONTEND_URL}/leases/${lease.id}`,
        uid: `lease-${lease.id}@stayspot.com`,
        sequence: 0,
        method: 'PUBLISH'
      };

      const calendar = ical.async;
      const icalString = calendar.toString(icalEvent);

      return {
        success: true,
        icalString,
        filename: `lease-${lease.id}.ics`
      };
    } catch (error) {
      console.error('iCal generation error:', error);
      return {
        success: false,
        error: error.message
      };
    }
  }

  // Generate iCal file for maintenance
  generateMaintenanceICal(maintenance) {
    try {
      if (!maintenance.scheduledDate) {
        throw new Error('Maintenance must have a scheduled date to generate iCal');
      }

      const icalEvent = {
        start: new Date(maintenance.scheduledDate),
        end: new Date(new Date(maintenance.scheduledDate).getTime() + 2 * 60 * 60 * 1000), // 2 hours
        summary: `Maintenance: ${maintenance.title}`,
        description: `Maintenance request for ${maintenance.property.address}. Priority: ${maintenance.priority}`,
        location: maintenance.property.address,
        url: `${process.env.FRONTEND_URL}/maintenance/${maintenance.id}`,
        uid: `maintenance-${maintenance.id}@stayspot.com`,
        sequence: 0,
        method: 'PUBLISH'
      };

      const calendar = ical.async;
      const icalString = calendar.toString(icalEvent);

      return {
        success: true,
        icalString,
        filename: `maintenance-${maintenance.id}.ics`
      };
    } catch (error) {
      console.error('Maintenance iCal generation error:', error);
      return {
        success: false,
        error: error.message
      };
    }
  }

  // Remove event from Google Calendar
  async removeFromGoogleCalendar(eventId, calendarId = 'primary') {
    try {
      if (!this.googleClient) {
        throw new Error('Google Calendar client not configured');
      }

      const authClient = await this.googleClient.getClient();
      
      await axios.delete(
        `https://www.googleapis.com/calendar/v3/calendars/${calendarId}/events/${eventId}`,
        {
          headers: {
            Authorization: `Bearer ${authClient.credentials.access_token}`
          }
        }
      );

      return {
        success: true,
        message: 'Event removed from Google Calendar successfully'
      };
    } catch (error) {
      console.error('Google Calendar removal error:', error);
      return {
        success: false,
        error: error.response?.data?.error?.message || error.message
      };
    }
  }

  // Update event in Google Calendar
  async updateGoogleCalendarEvent(eventId, updates, calendarId = 'primary') {
    try {
      if (!this.googleClient) {
        throw new Error('Google Calendar client not configured');
      }

      const authClient = await this.googleClient.getClient();
      
      const response = await axios.patch(
        `https://www.googleapis.com/calendar/v3/calendars/${calendarId}/events/${eventId}`,
        updates,
        {
          headers: {
            Authorization: `Bearer ${authClient.credentials.access_token}`,
            'Content-Type': 'application/json'
          }
        }
      );

      return {
        success: true,
        eventId: response.data.id,
        htmlLink: response.data.htmlLink,
        message: 'Google Calendar event updated successfully'
      };
    } catch (error) {
      console.error('Google Calendar update error:', error);
      return {
        success: false,
        error: error.response?.data?.error?.message || error.message
      };
    }
  }

  // Get calendar events for a date range
  async getCalendarEvents(startDate, endDate, calendarId = 'primary') {
    try {
      if (!this.googleClient) {
        throw new Error('Google Calendar client not configured');
      }

      const authClient = await this.googleClient.getClient();
      
      const response = await axios.get(
        `https://www.googleapis.com/calendar/v3/calendars/${calendarId}/events`,
        {
          params: {
            timeMin: new Date(startDate).toISOString(),
            timeMax: new Date(endDate).toISOString(),
            singleEvents: true,
            orderBy: 'startTime'
          },
          headers: {
            Authorization: `Bearer ${authClient.credentials.access_token}`
          }
        }
      );

      return {
        success: true,
        events: response.data.items,
        message: 'Calendar events retrieved successfully'
      };
    } catch (error) {
      console.error('Calendar events retrieval error:', error);
      return {
        success: false,
        error: error.response?.data?.error?.message || error.message
      };
    }
  }

  // Check calendar availability
  async checkAvailability(startDate, endDate, calendarId = 'primary') {
    try {
      const eventsResult = await this.getCalendarEvents(startDate, endDate, calendarId);
      
      if (!eventsResult.success) {
        return eventsResult;
      }

      const busySlots = eventsResult.events
        .filter(event => event.status !== 'cancelled')
        .map(event => ({
          start: event.start.dateTime || event.start.date,
          end: event.end.dateTime || event.end.date
        }));

      return {
        success: true,
        available: busySlots.length === 0,
        busySlots,
        message: busySlots.length === 0 ? 'Time slot is available' : 'Time slot has conflicts'
      };
    } catch (error) {
      console.error('Availability check error:', error);
      return {
        success: false,
        error: error.message
      };
    }
  }
}

module.exports = new CalendarSync();