const { User, Property, Lease, Payment, AuditLog } = require('../models');
const { Op } = require('sequelize');
const authConfig = require('../config/auth');

const userService = {
  // Get all users with filtering and pagination
  getUsers: async (filters = {}) => {
    try {
      const {
        page = 1,
        limit = 10,
        search,
        role,
        isActive,
        sortBy = 'createdAt',
        sortOrder = 'DESC'
      } = filters;

      const offset = (page - 1) * limit;
      const where = {};

      // Build search conditions
      if (search) {
        where[Op.or] = [
          { firstName: { [Op.like]: `%${search}%` } },
          { lastName: { [Op.like]: `%${search}%` } },
          { email: { [Op.like]: `%${search}%` } },
          { companyName: { [Op.like]: `%${search}%` } }
        ];
      }

      if (role) where.role = role;
      if (isActive !== undefined) where.isActive = isActive;

      const users = await User.findAndCountAll({
        where,
        attributes: { exclude: ['password'] },
        limit: parseInt(limit),
        offset: parseInt(offset),
        order: [[sortBy, sortOrder.toUpperCase()]]
      });

      return {
        success: true,
        data: {
          users: users.rows,
          pagination: {
            page: parseInt(page),
            limit: parseInt(limit),
            total: users.count,
            pages: Math.ceil(users.count / limit)
          }
        }
      };
    } catch (error) {
      console.error('User service - get users error:', error);
      throw error;
    }
  },

  // Get user by ID
  getUserById: async (userId, requesterId) => {
    try {
      const user = await User.findByPk(userId, {
        attributes: { exclude: ['password'] },
        include: [
          {
            model: Property,
            as: 'properties',
            attributes: ['id', 'title', 'address', 'status', 'isPublished'],
            required: false
          },
          {
            model: Lease,
            as: 'tenantLeases',
            include: [{
              model: Property,
              attributes: ['id', 'title', 'address']
            }],
            required: false
          }
        ]
      });

      if (!user) {
        throw new Error('User not found');
      }

      // Check if requester has permission to view this user
      const requester = await User.findByPk(requesterId);
      if (requesterId !== userId && !requester.role.includes('admin')) {
        // Remove sensitive information for non-admin viewers
        delete user.dataValues.settings;
        delete user.dataValues.licenseNumber;
      }

      return {
        success: true,
        data: { user }
      };
    } catch (error) {
      console.error('User service - get user error:', error);
      throw error;
    }
  },

  // Update user
  updateUser: async (userId, updateData, requesterId, ipAddress, userAgent) => {
    try {
      const user = await User.findByPk(userId);
      if (!user) {
        throw new Error('User not found');
      }

      const requester = await User.findByPk(requesterId);

      // Check permissions
      if (requesterId !== userId && !requester.role.includes('admin')) {
        throw new Error('Not authorized to update this user');
      }

      // Non-admins cannot update role or active status
      if (!requester.role.includes('admin')) {
        delete updateData.role;
        delete updateData.isActive;
        delete updateData.emailVerified;
      }

      // Store old values for audit
      const oldValues = { ...user.toJSON() };
      delete oldValues.password;

      await user.update(updateData);

      // Log the update
      await AuditLog.create({
        action: 'user_updated',
        entityType: 'user',
        entityId: user.id,
        userId: requesterId,
        affectedUserId: user.id,
        oldValues,
        newValues: user.toJSON(),
        description: `User ${user.email} updated by ${requester.email}`,
        ipAddress,
        userAgent
      });

      const updatedUser = await User.findByPk(userId, {
        attributes: { exclude: ['password'] }
      });

      return {
        success: true,
        data: { user: updatedUser }
      };
    } catch (error) {
      console.error('User service - update user error:', error);
      throw error;
    }
  },

  // Delete user
  deleteUser: async (userId, requesterId, ipAddress, userAgent) => {
    try {
      const user = await User.findByPk(userId);
      if (!user) {
        throw new Error('User not found');
      }

      // Prevent self-deletion
      if (requesterId === userId) {
        throw new Error('Cannot delete your own account');
      }

      const requester = await User.findByPk(requesterId);

      // Check if user has properties or active leases
      const propertyCount = await Property.count({ where: { landlordId: userId } });
      const activeLeaseCount = await Lease.count({ 
        where: { 
          [Op.or]: [
            { tenantId: userId, status: 'active' },
            { landlordId: userId, status: 'active' }
          ]
        }
      });

      if (propertyCount > 0 || activeLeaseCount > 0) {
        throw new Error('Cannot delete user with active properties or leases');
      }

      // Store user data for audit before deletion
      const userData = user.toJSON();
      delete userData.password;

      await user.destroy();

      // Log the deletion
      await AuditLog.create({
        action: 'user_deleted',
        entityType: 'user',
        entityId: userId,
        userId: requesterId,
        affectedUserId: userId,
        oldValues: userData,
        description: `User ${userData.email} deleted by ${requester.email}`,
        ipAddress,
        userAgent
      });

      return {
        success: true,
        message: 'User deleted successfully'
      };
    } catch (error) {
      console.error('User service - delete user error:', error);
      throw error;
    }
  },

  // Get user statistics
  getUserStats: async (userId, requesterId) => {
    try {
      const targetUserId = userId || requesterId;
      const user = await User.findByPk(targetUserId);
      
      if (!user) {
        throw new Error('User not found');
      }

      // Check permissions
      if (requesterId !== targetUserId && !requester.role.includes('admin')) {
        throw new Error('Not authorized to view these statistics');
      }

      let stats = {};

      switch (user.role) {
        case 'landlord':
        case 'bnb_host':
        case 'corporate_housing_manager':
        case 'student_housing_coordinator':
        case 'luxury_property_specialist':
          stats = await userService.getLandlordStats(targetUserId);
          break;
        
        case 'tenant':
          stats = await userService.getTenantStats(targetUserId);
          break;
        
        case 'property_manager':
        case 'regional_manager':
        case 'community_manager':
          stats = await userService.getManagerStats(targetUserId);
          break;
        
        case 'maintenance_supervisor':
        case 'maintenance_staff':
          stats = await userService.getMaintenanceStats(targetUserId);
          break;
        
        case 'financial_officer':
        case 'financial_analyst':
          stats = await userService.getFinancialStats(targetUserId);
          break;
        
        default:
          stats = await userService.getBasicStats(targetUserId);
      }

      return {
        success: true,
        data: { stats }
      };
    } catch (error) {
      console.error('User service - get user stats error:', error);
      throw error;
    }
  },

  // Landlord statistics
  getLandlordStats: async (userId) => {
    const propertyCount = await Property.count({ where: { landlordId: userId } });
    const activeLeaseCount = await Lease.count({ 
      where: { 
        landlordId: userId, 
        status: 'active' 
      } 
    });
    const vacantProperties = await Property.count({
      where: {
        landlordId: userId,
        status: 'available'
      }
    });

    const currentMonth = new Date();
    currentMonth.setDate(1);
    currentMonth.setHours(0, 0, 0, 0);

    const monthlyRevenue = await Payment.sum('amount', {
      where: {
        landlordId: userId,
        status: 'paid',
        paidDate: {
          [Op.gte]: currentMonth
        }
      }
    });

    const pendingPayments = await Payment.sum('amount', {
      where: {
        landlordId: userId,
        status: 'pending',
        dueDate: { [Op.lte]: new Date() }
      }
    });

    return {
      propertyCount,
      activeLeaseCount,
      vacantProperties,
      monthlyRevenue: monthlyRevenue || 0,
      pendingPayments: pendingPayments || 0,
      occupancyRate: propertyCount > 0 ? Math.round((activeLeaseCount / propertyCount) * 100) : 0
    };
  },

  // Tenant statistics
  getTenantStats: async (userId) => {
    const activeLeaseCount = await Lease.count({ 
      where: { 
        tenantId: userId, 
        status: 'active' 
      } 
    });
    const totalPaid = await Payment.sum('amount', {
      where: {
        tenantId: userId,
        status: 'paid'
      }
    });
    const maintenanceRequests = await Maintenance.count({
      where: { tenantId: userId }
    });
    const upcomingPayment = await Payment.findOne({
      where: {
        tenantId: userId,
        status: 'pending',
        dueDate: { [Op.gte]: new Date() }
      },
      order: [['dueDate', 'ASC']]
    });

    return {
      activeLeaseCount,
      totalPaid: totalPaid || 0,
      maintenanceRequests,
      upcomingPayment: upcomingPayment?.amount || 0
    };
  },

  // Manager statistics
  getManagerStats: async (userId) => {
    const managedProperties = await Property.count();
    const activeMaintenance = await Maintenance.count({
      where: { status: ['pending', 'in_progress'] }
    });
    const activeLeases = await Lease.count({ where: { status: 'active' } });
    const totalProperties = await Property.count();
    const occupancyRate = totalProperties > 0 ? Math.round((activeLeases / totalProperties) * 100) : 0;

    return {
      managedProperties,
      activeMaintenance,
      activeLeases,
      occupancyRate
    };
  },

  // Maintenance statistics
  getMaintenanceStats: async (userId) => {
    const assignedRequests = await Maintenance.count({ where: { assignedTo: userId } });
    const completedThisMonth = await Maintenance.count({
      where: {
        assignedTo: userId,
        status: 'completed',
        completedDate: {
          [Op.gte]: new Date(new Date().getFullYear(), new Date().getMonth(), 1)
        }
      }
    });
    const highPriorityRequests = await Maintenance.count({
      where: {
        assignedTo: userId,
        priority: ['high', 'emergency'],
        status: ['pending', 'in_progress']
      }
    });

    return {
      assignedRequests,
      completedThisMonth,
      highPriorityRequests
    };
  },

  // Financial statistics
  getFinancialStats: async (userId) => {
    const totalRevenue = await Payment.sum('amount', {
      where: { status: 'paid' }
    });
    const pendingRevenue = await Payment.sum('amount', {
      where: { status: 'pending' }
    });
    const processedThisMonth = await Payment.count({
      where: {
        status: 'paid',
        paidDate: {
          [Op.gte]: new Date(new Date().getFullYear(), new Date().getMonth(), 1)
        }
      }
    });

    return {
      totalRevenue: totalRevenue || 0,
      pendingRevenue: pendingRevenue || 0,
      processedThisMonth
    };
  },

  // Basic statistics for other roles
  getBasicStats: async (userId) => {
    const user = await User.findByPk(userId);
    return {
      role: user.role,
      memberSince: user.createdAt,
      lastLogin: user.lastLogin
    };
  },

  // Update user profile
  updateProfile: async (userId, profileData, ipAddress, userAgent) => {
    try {
      const user = await User.findByPk(userId);
      if (!user) {
        throw new Error('User not found');
      }

      const oldValues = { ...user.toJSON() };
      delete oldValues.password;

      await user.update(profileData);

      // Log the profile update
      await AuditLog.create({
        action: 'profile_updated',
        entityType: 'user',
        entityId: user.id,
        userId: user.id,
        oldValues,
        newValues: user.toJSON(),
        description: `Profile updated for ${user.email}`,
        ipAddress,
        userAgent
      });

      const updatedUser = await User.findByPk(userId, {
        attributes: { exclude: ['password'] }
      });

      return {
        success: true,
        data: { user: updatedUser }
      };
    } catch (error) {
      console.error('User service - update profile error:', error);
      throw error;
    }
  },

  // Upload user avatar
  uploadAvatar: async (userId, avatarUrl, ipAddress, userAgent) => {
    try {
      const user = await User.findByPk(userId);
      if (!user) {
        throw new Error('User not found');
      }

      const oldAvatar = user.avatar;
      await user.update({ avatar: avatarUrl });

      // Log the avatar update
      await AuditLog.create({
        action: 'avatar_updated',
        entityType: 'user',
        entityId: user.id,
        userId: user.id,
        oldValues: { avatar: oldAvatar },
        newValues: { avatar: avatarUrl },
        description: `Avatar updated for ${user.email}`,
        ipAddress,
        userAgent
      });

      return {
        success: true,
        data: { avatar: avatarUrl }
      };
    } catch (error) {
      console.error('User service - upload avatar error:', error);
      throw error;
    }
  },

  // Get users by role
  getUsersByRole: async (role, filters = {}) => {
    try {
      const { page = 1, limit = 50, isActive = true } = filters;
      const offset = (page - 1) * limit;

      const where = { role };
      if (isActive !== undefined) where.isActive = isActive;

      const users = await User.findAndCountAll({
        where,
        attributes: ['id', 'firstName', 'lastName', 'email', 'phone', 'avatar', 'companyName', 'isActive', 'createdAt'],
        limit: parseInt(limit),
        offset: parseInt(offset),
        order: [['createdAt', 'DESC']]
      });

      return {
        success: true,
        data: {
          users: users.rows,
          pagination: {
            page: parseInt(page),
            limit: parseInt(limit),
            total: users.count,
            pages: Math.ceil(users.count / limit)
          }
        }
      };
    } catch (error) {
      console.error('User service - get users by role error:', error);
      throw error;
    }
  }
};

module.exports = userService;