const { Property, User, Lease, Payment, Maintenance, Review } = require('../models');
const { Op } = require('sequelize');

const propertyService = {
  // Get all properties with filtering and pagination
  getProperties: async (filters = {}) => {
    try {
      const {
        page = 1,
        limit = 10,
        search,
        type,
        minPrice,
        maxPrice,
        bedrooms,
        bathrooms,
        city,
        state,
        status,
        landlordId,
        isPublished,
        sortBy = 'createdAt',
        sortOrder = 'DESC'
      } = filters;

      const offset = (page - 1) * limit;
      const where = {};

      // Build search conditions
      if (search) {
        where[Op.or] = [
          { title: { [Op.like]: `%${search}%` } },
          { description: { [Op.like]: `%${search}%` } },
          { address: { [Op.like]: `%${search}%` } },
          { city: { [Op.like]: `%${search}%` } }
        ];
      }

      // Build filter conditions
      if (type) where.type = type;
      if (status) where.status = status;
      if (landlordId) where.landlordId = landlordId;
      if (city) where.city = { [Op.like]: `%${city}%` };
      if (state) where.state = state;
      if (isPublished !== undefined) where.isPublished = isPublished;

      // Price range
      if (minPrice || maxPrice) {
        where.price = {};
        if (minPrice) where.price[Op.gte] = parseFloat(minPrice);
        if (maxPrice) where.price[Op.lte] = parseFloat(maxPrice);
      }

      // Bedrooms and bathrooms
      if (bedrooms) where.bedrooms = parseInt(bedrooms);
      if (bathrooms) where.bathrooms = parseFloat(bathrooms);

      const properties = await Property.findAndCountAll({
        where,
        include: [
          {
            model: User,
            as: 'landlord',
            attributes: ['id', 'firstName', 'lastName', 'email', 'phone', 'avatar', 'companyName']
          },
          {
            model: Lease,
            as: 'leases',
            where: { status: 'active' },
            required: false,
            include: [{
              model: User,
              as: 'tenant',
              attributes: ['id', 'firstName', 'lastName', 'email', 'phone']
            }]
          },
          {
            model: Review,
            as: 'reviews',
            where: { isPublished: true },
            required: false,
            attributes: ['rating']
          }
        ],
        limit: parseInt(limit),
        offset: parseInt(offset),
        order: [[sortBy, sortOrder.toUpperCase()]]
      });

      // Calculate average ratings
      const propertiesWithRatings = properties.rows.map(property => {
        const propertyJSON = property.toJSON();
        if (propertyJSON.reviews && propertyJSON.reviews.length > 0) {
          const totalRating = propertyJSON.reviews.reduce((sum, review) => sum + review.rating, 0);
          propertyJSON.averageRating = (totalRating / propertyJSON.reviews.length).toFixed(1);
        } else {
          propertyJSON.averageRating = null;
        }
        return propertyJSON;
      });

      return {
        success: true,
        data: {
          properties: propertiesWithRatings,
          pagination: {
            page: parseInt(page),
            limit: parseInt(limit),
            total: properties.count,
            pages: Math.ceil(properties.count / limit)
          }
        }
      };
    } catch (error) {
      console.error('Property service - get properties error:', error);
      throw error;
    }
  },

  // Get property by ID
  getPropertyById: async (propertyId, userId = null, userRole = null) => {
    try {
      const property = await Property.findByPk(propertyId, {
        include: [
          {
            model: User,
            as: 'landlord',
            attributes: { exclude: ['password'] }
          },
          {
            model: Lease,
            as: 'leases',
            include: [{
              model: User,
              as: 'tenant',
              attributes: ['id', 'firstName', 'lastName', 'email', 'phone']
            }]
          },
          {
            model: Maintenance,
            as: 'maintenances',
            limit: 5,
            order: [['createdAt', 'DESC']]
          },
          {
            model: Review,
            as: 'reviews',
            where: { isPublished: true },
            required: false,
            include: [{
              model: User,
              as: 'reviewer',
              attributes: ['id', 'firstName', 'lastName', 'avatar']
            }]
          }
        ]
      });

      if (!property) {
        throw new Error('Property not found');
      }

      // Check if user has permission to view this property
      const canView = await propertyService.canViewProperty(property, userId, userRole);
      if (!canView) {
        throw new Error('Not authorized to view this property');
      }

      // Calculate average rating
      const propertyJSON = property.toJSON();
      if (propertyJSON.reviews && propertyJSON.reviews.length > 0) {
        const totalRating = propertyJSON.reviews.reduce((sum, review) => sum + review.rating, 0);
        propertyJSON.averageRating = (totalRating / propertyJSON.reviews.length).toFixed(1);
        propertyJSON.reviewCount = propertyJSON.reviews.length;
      } else {
        propertyJSON.averageRating = null;
        propertyJSON.reviewCount = 0;
      }

      return {
        success: true,
        data: { property: propertyJSON }
      };
    } catch (error) {
      console.error('Property service - get property error:', error);
      throw error;
    }
  },

  // Create property
  createProperty: async (propertyData, userId, userRole, ipAddress, userAgent) => {
    try {
      // Check if user can create properties
      const canCreate = propertyService.canManageProperties(userRole);
      if (!canCreate) {
        throw new Error('Not authorized to create properties');
      }

      const property = await Property.create({
        ...propertyData,
        landlordId: userId
      });

      // Fetch created property with landlord info
      const createdProperty = await Property.findByPk(property.id, {
        include: [{
          model: User,
          as: 'landlord',
          attributes: ['id', 'firstName', 'lastName', 'email', 'phone']
        }]
      });

      // Log the creation
      await AuditLog.create({
        action: 'property_created',
        entityType: 'property',
        entityId: property.id,
        userId: userId,
        description: `Property "${property.title}" created by user ${userId}`,
        ipAddress,
        userAgent
      });

      return {
        success: true,
        data: { property: createdProperty }
      };
    } catch (error) {
      console.error('Property service - create property error:', error);
      throw error;
    }
  },

  // Update property
  updateProperty: async (propertyId, updateData, userId, userRole, ipAddress, userAgent) => {
    try {
      const property = await Property.findByPk(propertyId);
      if (!property) {
        throw new Error('Property not found');
      }

      // Check if user can update this property
      const canUpdate = await propertyService.canUpdateProperty(property, userId, userRole);
      if (!canUpdate) {
        throw new Error('Not authorized to update this property');
      }

      const oldValues = { ...property.toJSON() };

      await property.update(updateData);

      // Log the update
      await AuditLog.create({
        action: 'property_updated',
        entityType: 'property',
        entityId: property.id,
        userId: userId,
        oldValues,
        newValues: property.toJSON(),
        description: `Property "${property.title}" updated by user ${userId}`,
        ipAddress,
        userAgent
      });

      const updatedProperty = await Property.findByPk(propertyId, {
        include: [{
          model: User,
          as: 'landlord',
          attributes: ['id', 'firstName', 'lastName', 'email', 'phone']
        }]
      });

      return {
        success: true,
        data: { property: updatedProperty }
      };
    } catch (error) {
      console.error('Property service - update property error:', error);
      throw error;
    }
  },

  // Delete property
  deleteProperty: async (propertyId, userId, userRole, ipAddress, userAgent) => {
    try {
      const property = await Property.findByPk(propertyId);
      if (!property) {
        throw new Error('Property not found');
      }

      // Check if user can delete this property
      const canDelete = await propertyService.canDeleteProperty(property, userId, userRole);
      if (!canDelete) {
        throw new Error('Not authorized to delete this property');
      }

      // Check if property has active leases
      const activeLease = await Lease.findOne({
        where: {
          propertyId: propertyId,
          status: 'active'
        }
      });

      if (activeLease) {
        throw new Error('Cannot delete property with active lease');
      }

      const propertyData = property.toJSON();

      await property.destroy();

      // Log the deletion
      await AuditLog.create({
        action: 'property_deleted',
        entityType: 'property',
        entityId: propertyId,
        userId: userId,
        oldValues: propertyData,
        description: `Property "${propertyData.title}" deleted by user ${userId}`,
        ipAddress,
        userAgent
      });

      return {
        success: true,
        message: 'Property deleted successfully'
      };
    } catch (error) {
      console.error('Property service - delete property error:', error);
      throw error;
    }
  },

  // Search properties
  searchProperties: async (searchCriteria = {}) => {
    try {
      const {
        query,
        city,
        minPrice,
        maxPrice,
        bedrooms,
        bathrooms,
        type,
        amenities,
        page = 1,
        limit = 12
      } = searchCriteria;

      const offset = (page - 1) * limit;
      const where = {
        isPublished: true,
        status: 'available'
      };

      // Text search
      if (query) {
        where[Op.or] = [
          { title: { [Op.like]: `%${query}%` } },
          { description: { [Op.like]: `%${query}%` } },
          { address: { [Op.like]: `%${query}%` } },
          { city: { [Op.like]: `%${query}%` } }
        ];
      }

      // Location filter
      if (city) where.city = { [Op.like]: `%${city}%` };

      // Property type filter
      if (type) where.type = type;

      // Numeric filters
      if (bedrooms) where.bedrooms = parseInt(bedrooms);
      if (bathrooms) where.bathrooms = parseFloat(bathrooms);

      // Price range
      if (minPrice || maxPrice) {
        where.price = {};
        if (minPrice) where.price[Op.gte] = parseFloat(minPrice);
        if (maxPrice) where.price[Op.lte] = parseFloat(maxPrice);
      }

      // Amenities filter (if implemented in JSON field)
      if (amenities && amenities.length > 0) {
        where.amenities = {
          [Op.overlap]: amenities
        };
      }

      const properties = await Property.findAndCountAll({
        where,
        include: [
          {
            model: User,
            as: 'landlord',
            attributes: ['id', 'firstName', 'lastName', 'companyName', 'avatar']
          },
          {
            model: Review,
            as: 'reviews',
            where: { isPublished: true },
            required: false,
            attributes: ['rating']
          }
        ],
        order: [['createdAt', 'DESC']],
        limit: parseInt(limit),
        offset: parseInt(offset)
      });

      // Calculate average ratings
      const propertiesWithRatings = properties.rows.map(property => {
        const propertyJSON = property.toJSON();
        if (propertyJSON.reviews && propertyJSON.reviews.length > 0) {
          const totalRating = propertyJSON.reviews.reduce((sum, review) => sum + review.rating, 0);
          propertyJSON.averageRating = (totalRating / propertyJSON.reviews.length).toFixed(1);
        } else {
          propertyJSON.averageRating = null;
        }
        return propertyJSON;
      });

      return {
        success: true,
        data: {
          properties: propertiesWithRatings,
          pagination: {
            page: parseInt(page),
            limit: parseInt(limit),
            total: properties.count,
            pages: Math.ceil(properties.count / limit)
          }
        }
      };
    } catch (error) {
      console.error('Property service - search properties error:', error);
      throw error;
    }
  },

  // Get properties by landlord
  getPropertiesByLandlord: async (landlordId, filters = {}) => {
    try {
      const { page = 1, limit = 10, status } = filters;
      const offset = (page - 1) * limit;

      const where = { landlordId };
      if (status) where.status = status;

      const properties = await Property.findAndCountAll({
        where,
        include: [
          {
            model: Lease,
            as: 'leases',
            required: false,
            include: [{
              model: User,
              as: 'tenant',
              attributes: ['id', 'firstName', 'lastName', 'email']
            }]
          },
          {
            model: Maintenance,
            as: 'maintenances',
            where: { status: ['pending', 'in_progress'] },
            required: false
          }
        ],
        limit: parseInt(limit),
        offset: parseInt(offset),
        order: [['createdAt', 'DESC']]
      });

      return {
        success: true,
        data: {
          properties: properties.rows,
          pagination: {
            page: parseInt(page),
            limit: parseInt(limit),
            total: properties.count,
            pages: Math.ceil(properties.count / limit)
          }
        }
      };
    } catch (error) {
      console.error('Property service - get landlord properties error:', error);
      throw error;
    }
  },

  // Toggle property publish status
  togglePublish: async (propertyId, userId, userRole, ipAddress, userAgent) => {
    try {
      const property = await Property.findByPk(propertyId);
      if (!property) {
        throw new Error('Property not found');
      }

      // Check if user can update this property
      const canUpdate = await propertyService.canUpdateProperty(property, userId, userRole);
      if (!canUpdate) {
        throw new Error('Not authorized to update this property');
      }

      const newPublishStatus = !property.isPublished;
      await property.update({ isPublished: newPublishStatus });

      // Log the status change
      await AuditLog.create({
        action: 'property_publish_toggled',
        entityType: 'property',
        entityId: property.id,
        userId: userId,
        oldValues: { isPublished: property.isPublished },
        newValues: { isPublished: newPublishStatus },
        description: `Property "${property.title}" ${newPublishStatus ? 'published' : 'unpublished'} by user ${userId}`,
        ipAddress,
        userAgent
      });

      return {
        success: true,
        data: { 
          property,
          message: `Property ${newPublishStatus ? 'published' : 'unpublished'} successfully`
        }
      };
    } catch (error) {
      console.error('Property service - toggle publish error:', error);
      throw error;
    }
  },

  // Get property statistics
  getPropertyStats: async (propertyId, userId, userRole) => {
    try {
      const property = await Property.findByPk(propertyId);
      if (!property) {
        throw new Error('Property not found');
      }

      // Check if user can view property stats
      const canView = await propertyService.canViewProperty(property, userId, userRole);
      if (!canView) {
        throw new Error('Not authorized to view property statistics');
      }

      const activeLease = await Lease.findOne({
        where: {
          propertyId: propertyId,
          status: 'active'
        },
        include: [{
          model: User,
          as: 'tenant',
          attributes: ['id', 'firstName', 'lastName', 'email', 'phone']
        }]
      });

      const totalRevenue = await Payment.sum('amount', {
        where: {
          propertyId: propertyId,
          status: 'paid'
        }
      });

      const maintenanceCount = await Maintenance.count({
        where: { propertyId: propertyId }
      });

      const reviewStats = await Review.findOne({
        where: { propertyId: propertyId },
        attributes: [
          [Review.sequelize.fn('AVG', Review.sequelize.col('rating')), 'averageRating'],
          [Review.sequelize.fn('COUNT', Review.sequelize.col('id')), 'reviewCount']
        ]
      });

      return {
        success: true,
        data: {
          stats: {
            activeLease,
            totalRevenue: totalRevenue || 0,
            maintenanceCount,
            averageRating: reviewStats?.dataValues.averageRating || null,
            reviewCount: reviewStats?.dataValues.reviewCount || 0
          }
        }
      };
    } catch (error) {
      console.error('Property service - get property stats error:', error);
      throw error;
    }
  },

  // Permission check methods
  canManageProperties: (userRole) => {
    const allowedRoles = [
      'super_admin',
      'system_admin',
      'landlord',
      'bnb_host',
      'corporate_housing_manager',
      'student_housing_coordinator',
      'luxury_property_specialist',
      'property_manager'
    ];
    return allowedRoles.includes(userRole);
  },

  canViewProperty: async (property, userId, userRole) => {
    if (!property.isPublished && property.landlordId !== userId && !userRole.includes('admin')) {
      return false;
    }

    if (userRole.includes('admin') || userRole === 'landlord' || property.landlordId === userId) {
      return true;
    }

    // Check if user is a tenant of this property
    if (userRole === 'tenant') {
      const tenantLease = await Lease.findOne({
        where: {
          propertyId: property.id,
          tenantId: userId,
          status: 'active'
        }
      });
      return !!tenantLease;
    }

    // Staff roles with property access
    const staffRoles = [
      'property_manager',
      'property_agent',
      'regional_manager',
      'leasing_consultant',
      'maintenance_supervisor',
      'maintenance_staff',
      'financial_officer',
      'financial_analyst',
      'marketing_specialist',
      'legal_advisor',
      'insurance_coordinator',
      'relocation_specialist',
      'community_manager',
      'inspector',
      'vendor',
      'corporate_housing_manager',
      'student_housing_coordinator',
      'luxury_property_specialist',
      'data_analyst',
      'customer_support_agent'
    ];

    return staffRoles.includes(userRole);
  },

  canUpdateProperty: async (property, userId, userRole) => {
    if (userRole.includes('admin')) {
      return true;
    }

    if (property.landlordId === userId) {
      return true;
    }

    // Property managers can update properties they manage
    if (userRole === 'property_manager' || userRole === 'regional_manager') {
      return true;
    }

    return false;
  },

  canDeleteProperty: async (property, userId, userRole) => {
    if (userRole.includes('admin')) {
      return true;
    }

    if (property.landlordId === userId) {
      return true;
    }

    return false;
  }
};

module.exports = propertyService;