const { Notification, User } = require('../models');
const { Op } = require('sequelize');
const emailService = require('../utils/emailService');

const notificationService = {
  // Get user notifications
  getUserNotifications: async (userId, filters = {}) => {
    try {
      const { page = 1, limit = 20, unreadOnly } = filters;
      const offset = (page - 1) * limit;

      const where = { userId };
      if (unreadOnly === 'true') {
        where.isRead = false;
      }

      const notifications = await Notification.findAndCountAll({
        where,
        order: [['createdAt', 'DESC']],
        limit: parseInt(limit),
        offset: parseInt(offset)
      });

      // Get unread count
      const unreadCount = await Notification.count({
        where: { userId, isRead: false }
      });

      return {
        success: true,
        data: {
          notifications: notifications.rows,
          unreadCount,
          pagination: {
            page: parseInt(page),
            limit: parseInt(limit),
            total: notifications.count,
            pages: Math.ceil(notifications.count / limit)
          }
        }
      };
    } catch (error) {
      console.error('Notification service - get notifications error:', error);
      throw error;
    }
  },

  // Mark notification as read
  markAsRead: async (notificationId, userId) => {
    try {
      const notification = await Notification.findOne({
        where: { id: notificationId, userId }
      });

      if (!notification) {
        throw new Error('Notification not found');
      }

      await notification.markAsRead();

      return {
        success: true,
        data: { notification }
      };
    } catch (error) {
      console.error('Notification service - mark as read error:', error);
      throw error;
    }
  },

  // Mark all notifications as read
  markAllAsRead: async (userId) => {
    try {
      await Notification.update(
        { isRead: true },
        {
          where: {
            userId,
            isRead: false
          }
        }
      );

      const unreadCount = await Notification.count({
        where: { userId, isRead: false }
      });

      return {
        success: true,
        data: { unreadCount }
      };
    } catch (error) {
      console.error('Notification service - mark all as read error:', error);
      throw error;
    }
  },

  // Delete notification
  deleteNotification: async (notificationId, userId) => {
    try {
      const notification = await Notification.findOne({
        where: { id: notificationId, userId }
      });

      if (!notification) {
        throw new Error('Notification not found');
      }

      await notification.destroy();

      return {
        success: true
      };
    } catch (error) {
      console.error('Notification service - delete notification error:', error);
      throw error;
    }
  },

  // Clear all notifications
  clearAllNotifications: async (userId) => {
    try {
      await Notification.destroy({
        where: { userId }
      });

      return {
        success: true
      };
    } catch (error) {
      console.error('Notification service - clear all notifications error:', error);
      throw error;
    }
  },

  // Create notification
  createNotification: async (notificationData) => {
    try {
      const {
        userId,
        title,
        message,
        type = 'system_alert',
        priority = 'medium',
        metadata = {},
        relatedEntityType = null,
        relatedEntityId = null,
        expiresAt = null
      } = notificationData;

      const notification = await Notification.create({
        userId,
        title,
        message,
        type,
        priority,
        metadata,
        relatedEntityType,
        relatedEntityId,
        expiresAt
      });

      // Send email notification based on user preferences
      const user = await User.findByPk(userId);
      if (user.settings?.notifications?.email) {
        await emailService.sendNotificationEmail(user, notification);
      }

      return {
        success: true,
        data: { notification }
      };
    } catch (error) {
      console.error('Notification service - create notification error:', error);
      throw error;
    }
  },

  // Create multiple notifications
  createBulkNotifications: async (notificationsData) => {
    try {
      const notifications = await Notification.bulkCreate(notificationsData);

      // Send email notifications (in background)
      notifications.forEach(async (notification) => {
        try {
          const user = await User.findByPk(notification.userId);
          if (user?.settings?.notifications?.email) {
            await emailService.sendNotificationEmail(user, notification);
          }
        } catch (emailError) {
          console.error('Failed to send notification email:', emailError);
        }
      });

      return {
        success: true,
        data: { count: notifications.length }
      };
    } catch (error) {
      console.error('Notification service - create bulk notifications error:', error);
      throw error;
    }
  },

  // Send payment reminder notifications
  sendPaymentReminderNotifications: async (payment) => {
    try {
      const notifications = [];

      // Notify tenant
      notifications.push({
        userId: payment.tenantId,
        title: 'Payment Reminder',
        message: `Your rent payment of $${payment.amount} for ${payment.property.title} is due on ${new Date(payment.dueDate).toLocaleDateString()}.`,
        type: 'payment_reminder',
        priority: 'medium',
        metadata: {
          paymentId: payment.id,
          amount: payment.amount,
          dueDate: payment.dueDate,
          propertyId: payment.propertyId
        },
        relatedEntityType: 'payment',
        relatedEntityId: payment.id
      });

      await notificationService.createBulkNotifications(notifications);

      return {
        success: true,
        data: { notificationsSent: notifications.length }
      };
    } catch (error) {
      console.error('Notification service - payment reminder error:', error);
      throw error;
    }
  },

  // Send maintenance update notifications
  sendMaintenanceUpdateNotifications: async (maintenance, updateType) => {
    try {
      const notifications = [];
      const statusMessages = {
        assigned: 'has been assigned to our maintenance team',
        in_progress: 'is now in progress',
        completed: 'has been completed',
        cancelled: 'has been cancelled'
      };

      const message = `Your maintenance request "${maintenance.title}" ${statusMessages[updateType]}.`;

      // Notify tenant
      notifications.push({
        userId: maintenance.tenantId,
        title: 'Maintenance Update',
        message,
        type: 'maintenance_update',
        priority: 'medium',
        metadata: {
          maintenanceId: maintenance.id,
          propertyId: maintenance.propertyId,
          updateType
        },
        relatedEntityType: 'maintenance',
        relatedEntityId: maintenance.id
      });

      // Notify landlord for important updates
      if (updateType === 'completed' || updateType === 'cancelled') {
        notifications.push({
          userId: maintenance.landlordId,
          title: 'Maintenance Update',
          message: `Maintenance request "${maintenance.title}" for your property ${statusMessages[updateType]}.`,
          type: 'maintenance_update',
          priority: 'medium',
          metadata: {
            maintenanceId: maintenance.id,
            propertyId: maintenance.propertyId,
            updateType
          },
          relatedEntityType: 'maintenance',
          relatedEntityId: maintenance.id
        });
      }

      await notificationService.createBulkNotifications(notifications);

      return {
        success: true,
        data: { notificationsSent: notifications.length }
      };
    } catch (error) {
      console.error('Notification service - maintenance update error:', error);
      throw error;
    }
  },

  // Send lease expiry notifications
  sendLeaseExpiryNotifications: async (lease, daysUntilExpiry) => {
    try {
      const notifications = [];

      // Notify tenant
      notifications.push({
        userId: lease.tenantId,
        title: 'Lease Expiry Reminder',
        message: `Your lease for ${lease.property.title} will expire in ${daysUntilExpiry} days. Please contact your landlord to discuss renewal options.`,
        type: 'lease_expiry',
        priority: 'medium',
        metadata: {
          leaseId: lease.id,
          propertyId: lease.propertyId,
          expiryDate: lease.endDate,
          daysUntilExpiry
        },
        relatedEntityType: 'lease',
        relatedEntityId: lease.id
      });

      // Notify landlord
      notifications.push({
        userId: lease.landlordId,
        title: 'Lease Expiry Reminder',
        message: `The lease for ${lease.property.title} with tenant ${lease.tenant.firstName} ${lease.tenant.lastName} will expire in ${daysUntilExpiry} days.`,
        type: 'lease_expiry',
        priority: 'medium',
        metadata: {
          leaseId: lease.id,
          propertyId: lease.propertyId,
          tenantId: lease.tenantId,
          expiryDate: lease.endDate,
          daysUntilExpiry
        },
        relatedEntityType: 'lease',
        relatedEntityId: lease.id
      });

      await notificationService.createBulkNotifications(notifications);

      return {
        success: true,
        data: { notificationsSent: notifications.length }
      };
    } catch (error) {
      console.error('Notification service - lease expiry error:', error);
      throw error;
    }
  },

  // Get notification statistics
  getNotificationStats: async (userId) => {
    try {
      const totalNotifications = await Notification.count({ where: { userId } });
      const unreadCount = await Notification.count({ where: { userId, isRead: false } });
      
      const notificationTypes = await Notification.findAll({
        where: { userId },
        attributes: [
          'type',
          [Notification.sequelize.fn('COUNT', Notification.sequelize.col('id')), 'count']
        ],
        group: ['type']
      });

      const recentActivity = await Notification.findAll({
        where: { userId },
        order: [['createdAt', 'DESC']],
        limit: 5
      });

      return {
        success: true,
        data: {
          stats: {
            totalNotifications,
            unreadCount,
            notificationTypes,
            recentActivity
          }
        }
      };
    } catch (error) {
      console.error('Notification service - get stats error:', error);
      throw error;
    }
  },

  // Clean up expired notifications
  cleanupExpiredNotifications: async () => {
    try {
      const result = await Notification.destroy({
        where: {
          expiresAt: {
            [Op.lt]: new Date()
          }
        }
      });

      return {
        success: true,
        data: { deletedCount: result }
      };
    } catch (error) {
      console.error('Notification service - cleanup error:', error);
      throw error;
    }
  }
};

module.exports = notificationService;