const jwt = require('jsonwebtoken');
const bcrypt = require('bcryptjs');
const { User, AuditLog } = require('../models');
const authConfig = require('../config/auth');
const emailService = require('../utils/emailService');

const authService = {
  // Register new user
  registerUser: async (userData) => {
    try {
      const {
        email,
        password,
        firstName,
        lastName,
        phone,
        role,
        companyName,
        licenseNumber
      } = userData;

      // Check if user already exists
      const existingUser = await User.findOne({ where: { email } });
      if (existingUser) {
        throw new Error('User already exists with this email');
      }

      // Validate role
      const validRoles = Object.values(authConfig.roles);
      if (!validRoles.includes(role)) {
        throw new Error('Invalid user role');
      }

      // Create user
      const user = await User.create({
        email,
        password,
        firstName,
        lastName,
        phone,
        role,
        companyName,
        licenseNumber
      });

      // Generate tokens
      const token = jwt.sign(
        {
          userId: user.id,
          email: user.email,
          role: user.role
        },
        authConfig.jwt.secret,
        { expiresIn: authConfig.jwt.expiresIn }
      );

      const refreshToken = jwt.sign(
        {
          userId: user.id,
          type: 'refresh'
        },
        authConfig.jwt.refreshSecret,
        { expiresIn: authConfig.jwt.refreshExpiresIn }
      );

      // Send welcome email
      await emailService.sendWelcomeEmail(user);

      // Log the registration
      await AuditLog.create({
        action: 'user_registered',
        entityType: 'user',
        entityId: user.id,
        userId: user.id,
        description: `User ${user.email} registered with role ${user.role}`,
        ipAddress: userData.ipAddress,
        userAgent: userData.userAgent
      });

      return {
        success: true,
        user: user.toJSON(),
        token,
        refreshToken
      };
    } catch (error) {
      console.error('Auth service - register error:', error);
      throw error;
    }
  },

  // Login user
  loginUser: async (email, password, ipAddress, userAgent) => {
    try {
      // Find user
      const user = await User.findOne({ where: { email } });
      if (!user) {
        throw new Error('Invalid email or password');
      }

      // Check if user is active
      if (!user.isActive) {
        throw new Error('Account is deactivated. Please contact support.');
      }

      // Validate password
      const isValidPassword = await user.validatePassword(password);
      if (!isValidPassword) {
        throw new Error('Invalid email or password');
      }

      // Update last login
      user.lastLogin = new Date();
      await user.save();

      // Generate tokens
      const token = jwt.sign(
        {
          userId: user.id,
          email: user.email,
          role: user.role
        },
        authConfig.jwt.secret,
        { expiresIn: authConfig.jwt.expiresIn }
      );

      const refreshToken = jwt.sign(
        {
          userId: user.id,
          type: 'refresh'
        },
        authConfig.jwt.refreshSecret,
        { expiresIn: authConfig.jwt.refreshExpiresIn }
      );

      // Log the login
      await AuditLog.create({
        action: 'user_login',
        entityType: 'user',
        entityId: user.id,
        userId: user.id,
        description: `User ${user.email} logged in`,
        ipAddress,
        userAgent
      });

      return {
        success: true,
        user: user.toJSON(),
        token,
        refreshToken
      };
    } catch (error) {
      console.error('Auth service - login error:', error);
      throw error;
    }
  },

  // Refresh token
  refreshToken: async (refreshToken) => {
    try {
      if (!refreshToken) {
        throw new Error('Refresh token required');
      }

      const decoded = jwt.verify(refreshToken, authConfig.jwt.refreshSecret);
      const user = await User.findByPk(decoded.userId);

      if (!user || !user.isActive) {
        throw new Error('Invalid refresh token');
      }

      const newToken = jwt.sign(
        {
          userId: user.id,
          email: user.email,
          role: user.role
        },
        authConfig.jwt.secret,
        { expiresIn: authConfig.jwt.expiresIn }
      );

      const newRefreshToken = jwt.sign(
        {
          userId: user.id,
          type: 'refresh'
        },
        authConfig.jwt.refreshSecret,
        { expiresIn: authConfig.jwt.refreshExpiresIn }
      );

      return {
        success: true,
        token: newToken,
        refreshToken: newRefreshToken
      };
    } catch (error) {
      console.error('Auth service - refresh token error:', error);
      throw error;
    }
  },

  // Forgot password
  forgotPassword: async (email, ipAddress, userAgent) => {
    try {
      const user = await User.findOne({ where: { email } });
      if (!user) {
        // Don't reveal whether email exists
        return { success: true };
      }

      // Generate reset token (valid for 1 hour)
      const resetToken = jwt.sign(
        { userId: user.id, type: 'password_reset' },
        authConfig.jwt.secret,
        { expiresIn: '1h' }
      );

      const resetLink = `${process.env.FRONTEND_URL}/reset-password?token=${resetToken}`;

      // Send reset email
      await emailService.sendPasswordResetEmail(user, resetLink);

      // Log the request
      await AuditLog.create({
        action: 'password_reset_requested',
        entityType: 'user',
        entityId: user.id,
        userId: user.id,
        description: `Password reset requested for ${user.email}`,
        ipAddress,
        userAgent
      });

      return { success: true };
    } catch (error) {
      console.error('Auth service - forgot password error:', error);
      throw error;
    }
  },

  // Reset password
  resetPassword: async (token, newPassword, ipAddress, userAgent) => {
    try {
      const decoded = jwt.verify(token, authConfig.jwt.secret);
      if (decoded.type !== 'password_reset') {
        throw new Error('Invalid reset token');
      }

      const user = await User.findByPk(decoded.userId);
      if (!user) {
        throw new Error('Invalid reset token');
      }

      // Update password
      user.password = newPassword;
      await user.save();

      // Log the reset
      await AuditLog.create({
        action: 'password_reset',
        entityType: 'user',
        entityId: user.id,
        userId: user.id,
        description: `Password reset for ${user.email}`,
        ipAddress,
        userAgent
      });

      return { success: true };
    } catch (error) {
      console.error('Auth service - reset password error:', error);
      throw error;
    }
  },

  // Change password
  changePassword: async (userId, currentPassword, newPassword, ipAddress, userAgent) => {
    try {
      const user = await User.findByPk(userId);
      if (!user) {
        throw new Error('User not found');
      }

      // Verify current password
      const isValid = await user.validatePassword(currentPassword);
      if (!isValid) {
        throw new Error('Current password is incorrect');
      }

      // Update password
      user.password = newPassword;
      await user.save();

      // Log the change
      await AuditLog.create({
        action: 'password_changed',
        entityType: 'user',
        entityId: user.id,
        userId: user.id,
        description: `Password changed for ${user.email}`,
        ipAddress,
        userAgent
      });

      return { success: true };
    } catch (error) {
      console.error('Auth service - change password error:', error);
      throw error;
    }
  },

  // Verify token
  verifyToken: async (token) => {
    try {
      const decoded = jwt.verify(token, authConfig.jwt.secret);
      const user = await User.findByPk(decoded.userId, {
        attributes: { exclude: ['password'] }
      });

      if (!user || !user.isActive) {
        throw new Error('Invalid token');
      }

      return {
        success: true,
        user: user.toJSON()
      };
    } catch (error) {
      console.error('Auth service - verify token error:', error);
      throw error;
    }
  },

  // Get user permissions based on role
  getUserPermissions: (role) => {
    const permissions = {
      // Super Admin - Full access
      super_admin: {
        users: ['create', 'read', 'update', 'delete'],
        properties: ['create', 'read', 'update', 'delete'],
        leases: ['create', 'read', 'update', 'delete'],
        payments: ['create', 'read', 'update', 'delete'],
        maintenance: ['create', 'read', 'update', 'delete'],
        reports: ['create', 'read', 'update', 'delete'],
        system: ['manage']
      },
      // System Admin - System management
      system_admin: {
        users: ['create', 'read', 'update'],
        properties: ['create', 'read', 'update', 'delete'],
        leases: ['create', 'read', 'update', 'delete'],
        payments: ['create', 'read', 'update', 'delete'],
        maintenance: ['create', 'read', 'update', 'delete'],
        reports: ['create', 'read', 'update', 'delete'],
        system: ['manage']
      },
      // Landlord - Property management
      landlord: {
        properties: ['create', 'read', 'update', 'delete'],
        leases: ['create', 'read', 'update', 'delete'],
        payments: ['read', 'update'],
        maintenance: ['read', 'update'],
        reports: ['read']
      },
      // Tenant - Limited access
      tenant: {
        properties: ['read'],
        leases: ['read'],
        payments: ['create', 'read'],
        maintenance: ['create', 'read']
      },
      // Property Manager - Property operations
      property_manager: {
        properties: ['read', 'update'],
        leases: ['create', 'read', 'update'],
        payments: ['read'],
        maintenance: ['create', 'read', 'update'],
        reports: ['read']
      },
      // Property Agent - Leasing and showing
      property_agent: {
        properties: ['read'],
        leases: ['create', 'read', 'update'],
        payments: ['read'],
        maintenance: ['read']
      },
      // Regional Manager - Multi-property oversight
      regional_manager: {
        properties: ['read', 'update'],
        leases: ['read', 'update'],
        payments: ['read'],
        maintenance: ['read', 'update'],
        reports: ['read']
      },
      // Leasing Consultant - Lease management
      leasing_consultant: {
        properties: ['read'],
        leases: ['create', 'read', 'update'],
        payments: ['read']
      },
      // Maintenance Supervisor - Maintenance oversight
      maintenance_supervisor: {
        properties: ['read'],
        maintenance: ['create', 'read', 'update', 'delete'],
        reports: ['read']
      },
      // Maintenance Staff - Maintenance execution
      maintenance_staff: {
        properties: ['read'],
        maintenance: ['read', 'update']
      },
      // Financial Officer - Financial management
      financial_officer: {
        payments: ['create', 'read', 'update', 'delete'],
        reports: ['create', 'read', 'update', 'delete']
      },
      // Financial Analyst - Financial analysis
      financial_analyst: {
        payments: ['read'],
        reports: ['create', 'read']
      },
      // Marketing Specialist - Marketing operations
      marketing_specialist: {
        properties: ['read', 'update'],
        reports: ['read']
      },
      // Legal Advisor - Legal compliance
      legal_advisor: {
        leases: ['read', 'update'],
        properties: ['read'],
        reports: ['read']
      },
      // Insurance Coordinator - Insurance management
      insurance_coordinator: {
        properties: ['read'],
        leases: ['read'],
        reports: ['read']
      },
      // Relocation Specialist - Relocation services
      relocation_specialist: {
        properties: ['read'],
        leases: ['create', 'read']
      },
      // Community Manager - Community operations
      community_manager: {
        properties: ['read', 'update'],
        maintenance: ['read', 'update'],
        reports: ['read']
      },
      // Inspector - Property inspections
      inspector: {
        properties: ['read'],
        maintenance: ['create', 'read'],
        reports: ['create', 'read']
      },
      // Vendor - Service provider
      vendor: {
        maintenance: ['read', 'update']
      },
      // BNB Host - Short-term rental management
      bnb_host: {
        properties: ['create', 'read', 'update', 'delete'],
        leases: ['create', 'read', 'update', 'delete'],
        payments: ['read'],
        maintenance: ['create', 'read', 'update'],
        reports: ['read']
      },
      // Corporate Housing Manager - Corporate housing
      corporate_housing_manager: {
        properties: ['create', 'read', 'update'],
        leases: ['create', 'read', 'update'],
        payments: ['read'],
        maintenance: ['read', 'update'],
        reports: ['read']
      },
      // Student Housing Coordinator - Student housing
      student_housing_coordinator: {
        properties: ['create', 'read', 'update'],
        leases: ['create', 'read', 'update'],
        payments: ['read'],
        maintenance: ['read', 'update'],
        reports: ['read']
      },
      // Luxury Property Specialist - Luxury properties
      luxury_property_specialist: {
        properties: ['create', 'read', 'update'],
        leases: ['create', 'read', 'update'],
        payments: ['read'],
        maintenance: ['read', 'update'],
        reports: ['read']
      },
      // Data Analyst - Data analysis
      data_analyst: {
        properties: ['read'],
        leases: ['read'],
        payments: ['read'],
        maintenance: ['read'],
        reports: ['create', 'read', 'update']
      },
      // Customer Support Agent - Customer support
      customer_support_agent: {
        users: ['read'],
        properties: ['read'],
        leases: ['read'],
        payments: ['read'],
        maintenance: ['read', 'update']
      }
    };

    return permissions[role] || {};
  },

  // Check if user has permission
  hasPermission: (user, resource, action) => {
    const permissions = authService.getUserPermissions(user.role);
    return permissions[resource]?.includes(action) || false;
  }
};

module.exports = authService;