const { sequelize } = require('../config/database');
const { User, Property, Lease, Payment, Maintenance } = require('../models');
const bcrypt = require('bcryptjs');

const seedDatabase = async () => {
  try {
    console.log('Starting database seeding...');

    // Sync database
    await sequelize.sync({ force: true });
    console.log('Database synced successfully');

    // Create users
    const users = await User.bulkCreate([
      {
        email: 'superadmin@stayspot.com',
        password: await bcrypt.hash('password123', 12),
        firstName: 'Super',
        lastName: 'Admin',
        role: 'super_admin',
        isActive: true,
        emailVerified: true
      },
      {
        email: 'systemadmin@stayspot.com',
        password: await bcrypt.hash('password123', 12),
        firstName: 'System',
        lastName: 'Admin',
        role: 'system_admin',
        isActive: true,
        emailVerified: true
      },
      {
        email: 'landlord1@stayspot.com',
        password: await bcrypt.hash('password123', 12),
        firstName: 'John',
        lastName: 'Landlord',
        role: 'landlord',
        companyName: 'Premium Properties',
        isActive: true,
        emailVerified: true
      },
      {
        email: 'tenant1@stayspot.com',
        password: await bcrypt.hash('password123', 12),
        firstName: 'Alice',
        lastName: 'Tenant',
        role: 'tenant',
        isActive: true,
        emailVerified: true
      },
      {
        email: 'manager@stayspot.com',
        password: await bcrypt.hash('password123', 12),
        firstName: 'Bob',
        lastName: 'Manager',
        role: 'property_manager',
        isActive: true,
        emailVerified: true
      },
      {
        email: 'agent@stayspot.com',
        password: await bcrypt.hash('password123', 12),
        firstName: 'Carol',
        lastName: 'Agent',
        role: 'property_agent',
        isActive: true,
        emailVerified: true
      }
    ], { validate: true });
    console.log('Users created successfully');

    // Create properties
    const properties = await Property.bulkCreate([
      {
        title: 'Luxury Downtown Apartment',
        description: 'Beautiful luxury apartment in the heart of downtown with amazing city views.',
        type: 'apartment',
        price: 2500.00,
        bedrooms: 2,
        bathrooms: 2,
        area: 1200,
        address: '123 Main Street',
        city: 'New York',
        state: 'NY',
        zipCode: '10001',
        amenities: ['wifi', 'parking', 'gym', 'pool', 'air_conditioning'],
        status: 'available',
        isPublished: true,
        landlordId: users[2].id // landlord1
      },
      {
        title: 'Cozy Suburban House',
        description: 'Charming family home in quiet suburban neighborhood with large backyard.',
        type: 'house',
        price: 1800.00,
        bedrooms: 3,
        bathrooms: 2,
        area: 1800,
        address: '456 Oak Avenue',
        city: 'Chicago',
        state: 'IL',
        zipCode: '60007',
        amenities: ['parking', 'garden', 'laundry'],
        status: 'occupied',
        isPublished: true,
        landlordId: users[2].id // landlord1
      }
    ], { validate: true });
    console.log('Properties created successfully');

    // Create leases
    const leases = await Lease.bulkCreate([
      {
        startDate: new Date('2024-01-01'),
        endDate: new Date('2024-12-31'),
        monthlyRent: 1800.00,
        securityDeposit: 1800.00,
        status: 'active',
        propertyId: properties[1].id,
        tenantId: users[3].id, // tenant1
        landlordId: users[2].id // landlord1
      }
    ], { validate: true });
    console.log('Leases created successfully');

    // Create payments
    const payments = await Payment.bulkCreate([
      {
        amount: 1800.00,
        dueDate: new Date('2024-02-01'),
        paidDate: new Date('2024-01-28'),
        status: 'paid',
        paymentMethod: 'bank_transfer',
        description: 'Rent for February 2024',
        leaseId: leases[0].id,
        tenantId: users[3].id,
        landlordId: users[2].id,
        propertyId: properties[1].id
      },
      {
        amount: 1800.00,
        dueDate: new Date('2024-03-01'),
        status: 'pending',
        description: 'Rent for March 2024',
        leaseId: leases[0].id,
        tenantId: users[3].id,
        landlordId: users[2].id,
        propertyId: properties[1].id
      }
    ], { validate: true });
    console.log('Payments created successfully');

    // Create maintenance requests
    const maintenanceRequests = await Maintenance.bulkCreate([
      {
        title: 'Leaky Faucet in Kitchen',
        description: 'The kitchen faucet has been leaking for the past few days.',
        priority: 'medium',
        status: 'pending',
        category: 'plumbing',
        propertyId: properties[1].id,
        tenantId: users[3].id,
        landlordId: users[2].id
      }
    ], { validate: true });
    console.log('Maintenance requests created successfully');

    console.log('Database seeding completed successfully!');
    console.log('\nSample Login Credentials:');
    console.log('Super Admin: superadmin@stayspot.com / password123');
    console.log('Landlord: landlord1@stayspot.com / password123');
    console.log('Tenant: tenant1@stayspot.com / password123');

    process.exit(0);
  } catch (error) {
    console.error('Database seeding failed:', error);
    process.exit(1);
  }
};

// Run seeding if called directly
if (require.main === module) {
  seedDatabase();
}

module.exports = seedDatabase;