const { sequelize } = require('../config/database');
const { User, Property, Lease, Payment } = require('../models');
const bcrypt = require('bcryptjs');
const authConfig = require('../config/auth');

const seedDatabase = async () => {
  try {
    console.log('Starting database seeding...');

    // Create super admin
    const superAdmin = await User.create({
      email: 'admin@stayspot.com',
      password: 'admin123',
      firstName: 'Super',
      lastName: 'Admin',
      phone: '+1234567890',
      role: authConfig.roles.SUPER_ADMIN,
      emailVerified: true,
      isActive: true
    });

    // Create system admin
    const systemAdmin = await User.create({
      email: 'system@stayspot.com',
      password: 'system123',
      firstName: 'System',
      lastName: 'Admin',
      phone: '+1234567891',
      role: authConfig.roles.SYSTEM_ADMIN,
      emailVerified: true,
      isActive: true
    });

    // Create sample landlord
    const landlord = await User.create({
      email: 'landlord@example.com',
      password: 'landlord123',
      firstName: 'John',
      lastName: 'PropertyOwner',
      phone: '+1234567892',
      role: authConfig.roles.LANDLORD,
      emailVerified: true,
      isActive: true,
      companyName: 'Premium Properties LLC'
    });

    // Create sample tenant
    const tenant = await User.create({
      email: 'tenant@example.com',
      password: 'tenant123',
      firstName: 'Jane',
      lastName: 'Tenant',
      phone: '+1234567893',
      role: authConfig.roles.TENANT,
      emailVerified: true,
      isActive: true
    });

    // Create sample property manager
    const propertyManager = await User.create({
      email: 'manager@example.com',
      password: 'manager123',
      firstName: 'Mike',
      lastName: 'Manager',
      phone: '+1234567894',
      role: authConfig.roles.PROPERTY_MANAGER,
      emailVerified: true,
      isActive: true,
      companyName: 'Professional Property Management'
    });

    // Create sample properties
    const property1 = await Property.create({
      title: 'Luxury Downtown Apartment',
      description: 'Beautiful luxury apartment in the heart of downtown with amazing city views.',
      type: 'apartment',
      price: 2500.00,
      bedrooms: 2,
      bathrooms: 2,
      area: 1200,
      address: '123 Main Street',
      city: 'New York',
      state: 'NY',
      zipCode: '10001',
      amenities: ['wifi', 'parking', 'gym', 'pool', 'air_conditioning'],
      features: ['hardwood_floor', 'central_air', 'balcony'],
      landlordId: landlord.id,
      isPublished: true,
      status: 'available'
    });

    const property2 = await Property.create({
      title: 'Cozy Suburban House',
      description: 'Charming 3-bedroom house in quiet suburban neighborhood.',
      type: 'house',
      price: 1800.00,
      bedrooms: 3,
      bathrooms: 2,
      area: 1800,
      address: '456 Oak Avenue',
      city: 'Chicago',
      state: 'IL',
      zipCode: '60007',
      amenities: ['parking', 'garden', 'pet_friendly'],
      features: ['fireplace', 'patio', 'garage'],
      landlordId: landlord.id,
      isPublished: true,
      status: 'available'
    });

    // Create sample lease
    const lease = await Lease.create({
      startDate: new Date('2024-01-01'),
      endDate: new Date('2024-12-31'),
      monthlyRent: 2500.00,
      securityDeposit: 2500.00,
      status: 'active',
      propertyId: property1.id,
      tenantId: tenant.id,
      landlordId: landlord.id,
      signedAt: new Date('2023-12-15')
    });

    // Create sample payments
    await Payment.create({
      amount: 2500.00,
      dueDate: new Date('2024-01-01'),
      paidDate: new Date('2024-01-01'),
      status: 'paid',
      paymentMethod: 'bank_transfer',
      description: 'January 2024 Rent',
      leaseId: lease.id,
      tenantId: tenant.id,
      landlordId: landlord.id,
      propertyId: property1.id
    });

    await Payment.create({
      amount: 2500.00,
      dueDate: new Date('2024-02-01'),
      status: 'pending',
      description: 'February 2024 Rent',
      leaseId: lease.id,
      tenantId: tenant.id,
      landlordId: landlord.id,
      propertyId: property1.id
    });

    console.log('✅ Sample data created successfully!');
    console.log('Super Admin:', superAdmin.email);
    console.log('Landlord:', landlord.email);
    console.log('Tenant:', tenant.email);
    console.log('Properties:', await Property.count());
    console.log('Leases:', await Lease.count());
    console.log('Payments:', await Payment.count());

    process.exit(0);
  } catch (error) {
    console.error('❌ Database seeding failed:', error);
    process.exit(1);
  }
};

// Run seeding if called directly
if (require.main === module) {
  seedDatabase();
}

module.exports = seedDatabase;