const express = require('express');
const router = express.Router();

// In-memory storage for transportation data
let drivers = [
  { id: 1, name: 'John Kamau', vehicle: 'Pickup Truck', rating: 4.8, pricePerKm: 50, location: { lat: -0.3031, lng: 36.0800, address: 'Nakuru CBD' }, available: true },
  { id: 2, name: 'Mary Wanjiku', vehicle: 'Van', rating: 4.9, pricePerKm: 80, location: { lat: -0.3041, lng: 36.0810, address: 'Milimani' }, available: true },
  { id: 3, name: 'Peter Ochieng', vehicle: 'Lorry', rating: 4.7, pricePerKm: 120, location: { lat: -0.3021, lng: 36.0790, address: 'Pipeline' }, available: false }
];

let bookings = [];
let driverLocations = {};

// GET /api/transportation/drivers/nearby - Get nearby drivers
router.get('/drivers/nearby', async (req, res) => {
  try {
    const { lat, lng, radius = 10 } = req.query;
    
    // Filter available drivers within radius (simplified calculation)
    const nearbyDrivers = drivers.filter(driver => {
      if (!driver.available) return false;
      
      // Simple distance calculation (in real app, use proper geolocation)
      const distance = Math.sqrt(
        Math.pow(driver.location.lat - parseFloat(lat), 2) + 
        Math.pow(driver.location.lng - parseFloat(lng), 2)
      ) * 111; // Rough km conversion
      
      return distance <= radius;
    });

    res.json({
      success: true,
      data: nearbyDrivers
    });
  } catch (error) {
    res.status(500).json({
      success: false,
      message: 'Internal server error'
    });
  }
});

// POST /api/transportation/bookings - Create booking
router.post('/bookings', async (req, res) => {
  try {
    const { 
      userId, 
      driverId, 
      fromAddress, 
      toAddress, 
      vehicleType, 
      items, 
      moveDate,
      estimatedCost 
    } = req.body;

    const booking = {
      id: `BK${Date.now()}`,
      userId,
      driverId,
      fromAddress,
      toAddress,
      vehicleType,
      items,
      moveDate,
      estimatedCost,
      status: 'pending',
      createdAt: new Date().toISOString()
    };

    bookings.push(booking);

    // Update driver availability
    const driver = drivers.find(d => d.id === parseInt(driverId));
    if (driver) {
      driver.available = false;
    }

    res.status(201).json({
      success: true,
      data: booking
    });
  } catch (error) {
    res.status(500).json({
      success: false,
      message: 'Internal server error'
    });
  }
});

// GET /api/transportation/bookings/:id/track - Track booking
router.get('/bookings/:id/track', async (req, res) => {
  try {
    const { id } = req.params;
    const booking = bookings.find(b => b.id === id);
    
    if (!booking) {
      return res.status(404).json({
        success: false,
        message: 'Booking not found'
      });
    }

    const driver = drivers.find(d => d.id === booking.driverId);
    const currentLocation = driverLocations[booking.driverId] || driver?.location;

    res.json({
      success: true,
      data: {
        booking,
        driver,
        currentLocation,
        estimatedArrival: '15 mins', // Mock calculation
        status: 'in_transit'
      }
    });
  } catch (error) {
    res.status(500).json({
      success: false,
      message: 'Internal server error'
    });
  }
});

// PUT /api/transportation/drivers/:id/location - Update driver location
router.put('/drivers/:id/location', async (req, res) => {
  try {
    const { id } = req.params;
    const { lat, lng, address } = req.body;

    driverLocations[id] = {
      lat: parseFloat(lat),
      lng: parseFloat(lng),
      address,
      timestamp: new Date().toISOString()
    };

    // Update driver's location in main array
    const driver = drivers.find(d => d.id === parseInt(id));
    if (driver) {
      driver.location = driverLocations[id];
    }

    res.json({
      success: true,
      data: driverLocations[id]
    });
  } catch (error) {
    res.status(500).json({
      success: false,
      message: 'Internal server error'
    });
  }
});

// GET /api/transportation/bookings/driver/:driverId - Get driver's bookings
router.get('/bookings/driver/:driverId', async (req, res) => {
  try {
    const { driverId } = req.params;
    const driverBookings = bookings.filter(b => b.driverId === parseInt(driverId));

    res.json({
      success: true,
      data: driverBookings
    });
  } catch (error) {
    res.status(500).json({
      success: false,
      message: 'Internal server error'
    });
  }
});

module.exports = router;