const express = require('express');
const router = express.Router();
const reportController = require('../controllers/reportController');
const { authenticateToken } = require('../middleware/auth');
const { authorize } = require('../middleware/roleAuth');
const authConfig = require('../config/auth');

// Financial reports
router.get(
  '/financial',
  authenticateToken,
  authorize(
    authConfig.roles.LANDLORD,
    authConfig.roles.SUPER_ADMIN,
    authConfig.roles.SYSTEM_ADMIN,
    authConfig.roles.FINANCIAL_OFFICER
  ),
  reportController.generateFinancialReport
);

// Occupancy reports
router.get(
  '/occupancy',
  authenticateToken,
  authorize(
    authConfig.roles.LANDLORD,
    authConfig.roles.SUPER_ADMIN,
    authConfig.roles.SYSTEM_ADMIN,
    authConfig.roles.PROPERTY_MANAGER
  ),
  reportController.generateOccupancyReport
);

// Maintenance reports
router.get(
  '/maintenance',
  authenticateToken,
  authorize(
    authConfig.roles.LANDLORD,
    authConfig.roles.SUPER_ADMIN,
    authConfig.roles.SYSTEM_ADMIN,
    authConfig.roles.MAINTENANCE_SUPERVISOR
  ),
  reportController.generateMaintenanceReport
);

// Tenant reports
router.get(
  '/tenants',
  authenticateToken,
  authorize(
    authConfig.roles.LANDLORD,
    authConfig.roles.SUPER_ADMIN,
    authConfig.roles.SYSTEM_ADMIN,
    authConfig.roles.PROPERTY_MANAGER
  ),
  reportController.generateTenantReport
);

// Property performance report
router.get(
  '/property-performance',
  authenticateToken,
  authorize(
    authConfig.roles.LANDLORD,
    authConfig.roles.SUPER_ADMIN,
    authConfig.roles.SYSTEM_ADMIN
  ),
  reportController.generatePropertyPerformanceReport
);

// Lease expiry report
router.get(
  '/lease-expiry',
  authenticateToken,
  authorize(
    authConfig.roles.LANDLORD,
    authConfig.roles.SUPER_ADMIN,
    authConfig.roles.SYSTEM_ADMIN,
    authConfig.roles.PROPERTY_MANAGER
  ),
  reportController.generateLeaseExpiryReport
);

// Payment overdue report
router.get(
  '/payment-overdue',
  authenticateToken,
  authorize(
    authConfig.roles.LANDLORD,
    authConfig.roles.SUPER_ADMIN,
    authConfig.roles.SYSTEM_ADMIN,
    authConfig.roles.FINANCIAL_OFFICER
  ),
  reportController.generatePaymentOverdueReport
);

// Custom report generator
router.post(
  '/custom',
  authenticateToken,
  authorize(
    authConfig.roles.SUPER_ADMIN,
    authConfig.roles.SYSTEM_ADMIN
  ),
  reportController.generateCustomReport
);

// Export report data
router.get(
  '/export/:format',
  authenticateToken,
  reportController.exportReportData
);

module.exports = router;