const express = require('express');
const router = express.Router();
const maintenanceController = require('../controllers/maintenanceController');
const { authenticateToken } = require('../middleware/auth');
const { authorize } = require('../middleware/roleAuth');
const { validateId } = require('../middleware/validation');
const { handleUpload, uploadToCloudinaryMiddleware, setUploadFolder } = require('../middleware/upload');
const authConfig = require('../config/auth');

// Get all maintenance requests (filtered by user role)
router.get(
  '/',
  authenticateToken,
  maintenanceController.getAllMaintenance
);

// Get maintenance by ID
router.get(
  '/:id',
  authenticateToken,
  validateId,
  maintenanceController.getMaintenanceById
);

// Create maintenance request
router.post(
  '/',
  authenticateToken,
  authorize(authConfig.roles.TENANT),
  handleUpload,
  setUploadFolder('maintenance'),
  uploadToCloudinaryMiddleware,
  maintenanceController.createMaintenance
);

// Update maintenance request
router.put(
  '/:id',
  authenticateToken,
  validateId,
  maintenanceController.updateMaintenance
);

// Assign maintenance staff
router.patch(
  '/:id/assign',
  authenticateToken,
  validateId,
  authorize(
    authConfig.roles.LANDLORD,
    authConfig.roles.SUPER_ADMIN,
    authConfig.roles.SYSTEM_ADMIN,
    authConfig.roles.MAINTENANCE_SUPERVISOR
  ),
  maintenanceController.assignMaintenance
);

// Update maintenance status
router.patch(
  '/:id/status',
  authenticateToken,
  validateId,
  maintenanceController.updateStatus
);

// Add maintenance notes
router.patch(
  '/:id/notes',
  authenticateToken,
  validateId,
  maintenanceController.addNotes
);

// Complete maintenance
router.patch(
  '/:id/complete',
  authenticateToken,
  validateId,
  maintenanceController.completeMaintenance
);

// Cancel maintenance
router.patch(
  '/:id/cancel',
  authenticateToken,
  validateId,
  maintenanceController.cancelMaintenance
);

// Get maintenance by tenant
router.get(
  '/tenant/my-requests',
  authenticateToken,
  authorize(authConfig.roles.TENANT),
  (req, res) => {
    req.query.tenantId = req.user.id;
    maintenanceController.getAllMaintenance(req, res);
  }
);

// Get maintenance by property
router.get(
  '/property/:propertyId',
  authenticateToken,
  validateId,
  maintenanceController.getMaintenanceByProperty
);

// Get maintenance by assigned staff
router.get(
  '/assigned/my-assignments',
  authenticateToken,
  authorize(
    authConfig.roles.MAINTENANCE_STAFF,
    authConfig.roles.MAINTENANCE_SUPERVISOR
  ),
  (req, res) => {
    req.query.assignedTo = req.user.id;
    maintenanceController.getAllMaintenance(req, res);
  }
);

// Get maintenance statistics
router.get(
  '/stats/overview',
  authenticateToken,
  authorize(
    authConfig.roles.LANDLORD,
    authConfig.roles.SUPER_ADMIN,
    authConfig.roles.SYSTEM_ADMIN,
    authConfig.roles.MAINTENANCE_SUPERVISOR
  ),
  maintenanceController.getMaintenanceStats
);

module.exports = router;