const express = require('express');
const router = express.Router();
const leaseController = require('../controllers/leaseController');
const { authenticateToken } = require('../middleware/auth');
const { authorize } = require('../middleware/roleAuth');
const { validateId } = require('../middleware/validation');
const authConfig = require('../config/auth');

// Get all leases (filtered by user role)
router.get(
  '/',
  authenticateToken,
  leaseController.getAllLeases
);

// Get lease by ID
router.get(
  '/:id',
  authenticateToken,
  validateId,
  leaseController.getLeaseById
);

// Create new lease
router.post(
  '/',
  authenticateToken,
  authorize(
    authConfig.roles.LANDLORD,
    authConfig.roles.SUPER_ADMIN,
    authConfig.roles.SYSTEM_ADMIN,
    authConfig.roles.PROPERTY_MANAGER
  ),
  leaseController.createLease
);

// Update lease
router.put(
  '/:id',
  authenticateToken,
  validateId,
  leaseController.updateLease
);

// Terminate lease
router.patch(
  '/:id/terminate',
  authenticateToken,
  validateId,
  leaseController.terminateLease
);

// Renew lease
router.patch(
  '/:id/renew',
  authenticateToken,
  validateId,
  leaseController.renewLease
);

// Get lease payments
router.get(
  '/:id/payments',
  authenticateToken,
  validateId,
  leaseController.getLeasePayments
);

// Get leases by tenant
router.get(
  '/tenant/my-leases',
  authenticateToken,
  authorize(authConfig.roles.TENANT),
  (req, res) => {
    req.query.tenantId = req.user.id;
    leaseController.getAllLeases(req, res);
  }
);

// Get leases by landlord
router.get(
  '/landlord/my-leases',
  authenticateToken,
  authorize(
    authConfig.roles.LANDLORD,
    authConfig.roles.SUPER_ADMIN,
    authConfig.roles.SYSTEM_ADMIN
  ),
  (req, res) => {
    req.query.landlordId = req.user.id;
    leaseController.getAllLeases(req, res);
  }
);

module.exports = router;