const express = require('express');
const router = express.Router();

// In-memory storage for chat messages (replace with database)
let chatRooms = {};
let connectedUsers = {};

// GET /api/chat/messages/:roomId - Get chat messages for a room
router.get('/messages/:roomId', async (req, res) => {
  try {
    const { roomId } = req.params;
    const messages = chatRooms[roomId] || [];
    
    res.json({
      success: true,
      data: messages
    });
  } catch (error) {
    console.error('Error fetching messages:', error);
    res.status(500).json({
      success: false,
      message: 'Internal server error'
    });
  }
});

// POST /api/chat/messages - Send a message
router.post('/messages', async (req, res) => {
  try {
    const { roomId, message, userId, userName, userRole } = req.body;

    if (!roomId || !message || !userId) {
      return res.status(400).json({
        success: false,
        message: 'Missing required fields'
      });
    }

    const newMessage = {
      id: Date.now().toString(),
      roomId,
      message,
      userId,
      userName: userName || 'User',
      userRole: userRole || 'user',
      timestamp: new Date().toISOString(),
      sender: 'user'
    };

    if (!chatRooms[roomId]) {
      chatRooms[roomId] = [];
    }

    chatRooms[roomId].push(newMessage);

    // Simulate support response for demo
    setTimeout(() => {
      const supportMessage = {
        id: (Date.now() + 1).toString(),
        roomId,
        message: "Thank you for your message! Our support team is here to help.",
        userId: 'support',
        userName: 'StaySpot Support',
        userRole: 'support',
        timestamp: new Date().toISOString(),
        sender: 'support'
      };
      chatRooms[roomId].push(supportMessage);
    }, 1000);

    res.status(201).json({
      success: true,
      data: newMessage
    });

  } catch (error) {
    console.error('Error sending message:', error);
    res.status(500).json({
      success: false,
      message: 'Internal server error'
    });
  }
});

// GET /api/chat/users/online - Get online users count
router.get('/users/online', async (req, res) => {
  try {
    const onlineCount = Object.keys(connectedUsers).length;
    
    res.json({
      success: true,
      data: {
        onlineUsers: onlineCount,
        users: Object.values(connectedUsers)
      }
    });
  } catch (error) {
    console.error('Error fetching online users:', error);
    res.status(500).json({
      success: false,
      message: 'Internal server error'
    });
  }
});

module.exports = router;