const { DataTypes } = require('sequelize');
const { sequelize } = require('../config/database');
const bcrypt = require('bcryptjs');

// Define roles here to avoid circular dependency
const USER_ROLES = {
  SUPER_ADMIN: 'super_admin',
  SYSTEM_ADMIN: 'system_admin',
  LANDLORD: 'landlord',
  TENANT: 'tenant',
  AGENT: 'agent',
  PROPERTY_MANAGER: 'property_manager',
  REGIONAL_MANAGER: 'regional_manager',
  CORPORATE_HOUSING_MANAGER: 'corporate_housing_manager',
  STUDENT_HOUSING_COORDINATOR: 'student_housing_coordinator',
  COMMUNITY_MANAGER: 'community_manager',
  LUXURY_PROPERTY_SPECIALIST: 'luxury_property_specialist',
  LEASING_CONSULTANT: 'leasing_consultant',
  MAINTENANCE_SUPERVISOR: 'maintenance_supervisor',
  MAINTENANCE_STAFF: 'maintenance_staff',
  FINANCIAL_OFFICER: 'financial_officer',
  FINANCIAL_ANALYST: 'financial_analyst',
  MARKETING_SPECIALIST: 'marketing_specialist',
  LEGAL_ADVISOR: 'legal_advisor',
  INSURANCE_COORDINATOR: 'insurance_coordinator',
  RELOCATION_SPECIALIST: 'relocation_specialist',
  INSPECTOR: 'inspector',
  VENDOR: 'vendor',
  BNB_HOST: 'bnb_host',
  DATA_ANALYST: 'data_analyst',
  CUSTOMER_SUPPORT_AGENT: 'customer_support_agent'
};

const User = sequelize.define('User', {
  id: {
    type: DataTypes.INTEGER,
    primaryKey: true,
    autoIncrement: true
  },
  email: {
    type: DataTypes.STRING(255),
    allowNull: false,
    unique: true,
    validate: {
      isEmail: true
    }
  },
  password: {
    type: DataTypes.STRING(255),
    allowNull: false
  },
  firstName: {
    type: DataTypes.STRING(100),
    allowNull: false,
    field: 'first_name'
  },
  lastName: {
    type: DataTypes.STRING(100),
    allowNull: false,
    field: 'last_name'
  },
  phone: {
    type: DataTypes.STRING(20),
    allowNull: true
  },
  avatar: {
    type: DataTypes.STRING(500),
    allowNull: true
  },
  role: {
    type: DataTypes.ENUM(...Object.values(USER_ROLES)),
    allowNull: false,
    defaultValue: USER_ROLES.TENANT
  },
  isActive: {
    type: DataTypes.BOOLEAN,
    allowNull: false,
    defaultValue: true,
    field: 'is_active'
  },
  emailVerified: {
    type: DataTypes.BOOLEAN,
    allowNull: false,
    defaultValue: false,
    field: 'email_verified'
  },
  lastLogin: {
    type: DataTypes.DATE,
    allowNull: true,
    field: 'last_login'
  },
  companyName: {
    type: DataTypes.STRING(255),
    allowNull: true,
    field: 'company_name'
  },
  licenseNumber: {
    type: DataTypes.STRING(100),
    allowNull: true,
    field: 'license_number'
  },
  bio: {
    type: DataTypes.TEXT,
    allowNull: true
  },
  address: {
    type: DataTypes.JSON,
    allowNull: true
  },
  settings: {
    type: DataTypes.JSON,
    allowNull: true,
    defaultValue: {}
  }
}, {
  tableName: 'users',
  timestamps: true,
  hooks: {
    beforeCreate: async (user) => {
      if (user.password) {
        user.password = await bcrypt.hash(user.password, 12);
      }
    },
    beforeUpdate: async (user) => {
      if (user.changed('password')) {
        user.password = await bcrypt.hash(user.password, 12);
      }
    }
  }
});

// Instance methods
User.prototype.validatePassword = async function(password) {
  return await bcrypt.compare(password, this.password);
};

User.prototype.toJSON = function() {
  const values = { ...this.get() };
  delete values.password;
  return values;
};

// Static methods
User.findByEmail = function(email) {
  return this.findOne({ where: { email } });
};

User.getLandlords = function() {
  return this.findAll({ 
    where: { 
      role: USER_ROLES.LANDLORD,
      isActive: true 
    } 
  });
};

// Export roles for use in other files
User.ROLES = USER_ROLES;

module.exports = User;