const { DataTypes } = require('sequelize');
const { sequelize } = require('../config/database');

const Review = sequelize.define('Review', {
  id: {
    type: DataTypes.INTEGER,
    primaryKey: true,
    autoIncrement: true
  },
  rating: {
    type: DataTypes.INTEGER,
    allowNull: false,
    validate: {
      min: 1,
      max: 5
    }
  },
  title: {
    type: DataTypes.STRING(255),
    allowNull: true
  },
  comment: {
    type: DataTypes.TEXT,
    allowNull: true
  },
  reviewType: {
    type: DataTypes.ENUM('property', 'landlord', 'tenant'),
    allowNull: false,
    field: 'review_type'
  },
  response: {
    type: DataTypes.TEXT,
    allowNull: true
  },
  respondedAt: {
    type: DataTypes.DATE,
    allowNull: true,
    field: 'responded_at'
  },
  isPublished: {
    type: DataTypes.BOOLEAN,
    allowNull: false,
    defaultValue: true,
    field: 'is_published'
  },
  helpfulCount: {
    type: DataTypes.INTEGER,
    allowNull: false,
    defaultValue: 0,
    field: 'helpful_count'
  },
  reviewerId: {
    type: DataTypes.INTEGER,
    allowNull: false,
    field: 'reviewer_id',
    references: {
      model: 'users',
      key: 'id'
    }
  },
  revieweeId: {
    type: DataTypes.INTEGER,
    allowNull: true,
    field: 'reviewee_id',
    references: {
      model: 'users',
      key: 'id'
    }
  },
  propertyId: {
    type: DataTypes.INTEGER,
    allowNull: true,
    field: 'property_id',
    references: {
      model: 'properties',
      key: 'id'
    }
  },
  leaseId: {
    type: DataTypes.INTEGER,
    allowNull: true,
    field: 'lease_id',
    references: {
      model: 'leases',
      key: 'id'
    }
  }
}, {
  tableName: 'reviews',
  timestamps: true,
  indexes: [
    {
      fields: ['review_type', 'property_id']
    },
    {
      fields: ['reviewer_id']
    },
    {
      fields: ['rating']
    }
  ]
});

// Instance methods
Review.prototype.canRespond = function(userId) {
  if (this.reviewType === 'property' && this.property) {
    return this.property.landlordId === userId;
  }
  if (this.reviewType === 'landlord') {
    return this.revieweeId === userId;
  }
  return false;
};

Review.prototype.markHelpful = function() {
  this.helpfulCount += 1;
  return this.save();
};

// Static methods
Review.getPropertyReviews = function(propertyId) {
  return this.findAll({
    where: {
      propertyId,
      isPublished: true,
      reviewType: 'property'
    },
    include: ['reviewer'],
    order: [['createdAt', 'DESC']]
  });
};

Review.getUserReviews = function(userId, reviewType = null) {
  const where = reviewType ? { revieweeId: userId, reviewType } : { revieweeId: userId };
  return this.findAll({
    where,
    include: ['reviewer', 'property'],
    order: [['createdAt', 'DESC']]
  });
};

Review.getAverageRating = function(propertyId) {
  return this.findOne({
    where: { propertyId, reviewType: 'property' },
    attributes: [
      [sequelize.fn('AVG', sequelize.col('rating')), 'averageRating'],
      [sequelize.fn('COUNT', sequelize.col('id')), 'reviewCount']
    ]
  });
};

module.exports = Review;