const { DataTypes } = require('sequelize');
const { sequelize } = require('../config/database');

const Message = sequelize.define('Message', {
  id: {
    type: DataTypes.INTEGER,
    primaryKey: true,
    autoIncrement: true
  },
  content: {
    type: DataTypes.TEXT,
    allowNull: false
  },
  messageType: {
    type: DataTypes.ENUM('text', 'image', 'file', 'system'),
    allowNull: false,
    defaultValue: 'text',
    field: 'message_type'
  },
  attachments: {
    type: DataTypes.JSON,
    allowNull: true,
    defaultValue: []
  },
  isRead: {
    type: DataTypes.BOOLEAN,
    allowNull: false,
    defaultValue: false,
    field: 'is_read'
  },
  readAt: {
    type: DataTypes.DATE,
    allowNull: true,
    field: 'read_at'
  },
  senderId: {
    type: DataTypes.INTEGER,
    allowNull: false,
    field: 'sender_id',
    references: {
      model: 'users',
      key: 'id'
    }
  },
  receiverId: {
    type: DataTypes.INTEGER,
    allowNull: false,
    field: 'receiver_id',
    references: {
      model: 'users',
      key: 'id'
    }
  },
  propertyId: {
    type: DataTypes.INTEGER,
    allowNull: true,
    field: 'property_id',
    references: {
      model: 'properties',
      key: 'id'
    }
  },
  leaseId: {
    type: DataTypes.INTEGER,
    allowNull: true,
    field: 'lease_id',
    references: {
      model: 'leases',
      key: 'id'
    }
  },
  parentMessageId: {
    type: DataTypes.INTEGER,
    allowNull: true,
    field: 'parent_message_id',
    references: {
      model: 'messages',
      key: 'id'
    }
  },
  conversationId: {
    type: DataTypes.STRING(100),
    allowNull: false
  }
}, {
  tableName: 'messages',
  timestamps: true,
  indexes: [
    {
      fields: ['conversation_id']
    },
    {
      fields: ['sender_id', 'receiver_id']
    },
    {
      fields: ['created_at']
    }
  ]
});

// Instance methods
Message.prototype.markAsRead = function() {
  this.isRead = true;
  this.readAt = new Date();
  return this.save();
};

Message.prototype.getConversationId = function(senderId, receiverId, propertyId = null) {
  const ids = [senderId, receiverId].sort((a, b) => a - b);
  let conversationId = `conv_${ids[0]}_${ids[1]}`;
  if (propertyId) {
    conversationId += `_prop_${propertyId}`;
  }
  return conversationId;
};

// Static methods
Message.getConversation = function(user1Id, user2Id, propertyId = null, limit = 50) {
  const conversationId = Message.prototype.getConversationId(user1Id, user2Id, propertyId);
  
  return this.findAll({
    where: { conversationId },
    include: ['sender', 'receiver'],
    order: [['createdAt', 'DESC']],
    limit
  });
};

Message.getUnreadCount = function(userId) {
  return this.count({
    where: {
      receiverId: userId,
      isRead: false
    }
  });
};

Message.getUserConversations = function(userId) {
  return this.sequelize.query(`
    SELECT DISTINCT 
      m.conversation_id,
      CASE 
        WHEN m.sender_id = :userId THEN m.receiver_id 
        ELSE m.sender_id 
      END as other_user_id,
      MAX(m.created_at) as last_message_date
    FROM messages m
    WHERE m.sender_id = :userId OR m.receiver_id = :userId
    GROUP BY m.conversation_id, other_user_id
    ORDER BY last_message_date DESC
  `, {
    replacements: { userId },
    type: this.sequelize.QueryTypes.SELECT
  });
};

module.exports = Message;