const { DataTypes } = require('sequelize');
const { sequelize } = require('../config/database');

const AuditLog = sequelize.define('AuditLog', {
  id: {
    type: DataTypes.INTEGER,
    primaryKey: true,
    autoIncrement: true
  },
  action: {
    type: DataTypes.STRING(100),
    allowNull: false
  },
  entityType: {
    type: DataTypes.ENUM('user', 'property', 'lease', 'payment', 'maintenance', 'document', 'review'),
    allowNull: false,
    field: 'entity_type'
  },
  entityId: {
    type: DataTypes.INTEGER,
    allowNull: true,
    field: 'entity_id'
  },
  oldValues: {
    type: DataTypes.JSON,
    allowNull: true,
    field: 'old_values'
  },
  newValues: {
    type: DataTypes.JSON,
    allowNull: true,
    field: 'new_values'
  },
  description: {
    type: DataTypes.TEXT,
    allowNull: true
  },
  ipAddress: {
    type: DataTypes.STRING(45),
    allowNull: true,
    field: 'ip_address'
  },
  userAgent: {
    type: DataTypes.TEXT,
    allowNull: true,
    field: 'user_agent'
  },
  userId: {
    type: DataTypes.INTEGER,
    allowNull: false,
    field: 'user_id',
    references: {
      model: 'users',
      key: 'id'
    }
  },
  affectedUserId: {
    type: DataTypes.INTEGER,
    allowNull: true,
    field: 'affected_user_id',
    references: {
      model: 'users',
      key: 'id'
    }
  }
}, {
  tableName: 'audit_logs',
  timestamps: true,
  indexes: [
    {
      fields: ['entity_type', 'entity_id']
    },
    {
      fields: ['user_id']
    },
    {
      fields: ['created_at']
    },
    {
      fields: ['action']
    }
  ]
});

// Static methods
AuditLog.logAction = function(data) {
  return this.create(data);
};

AuditLog.getEntityHistory = function(entityType, entityId, limit = 50) {
  return this.findAll({
    where: { entityType, entityId },
    include: ['user'],
    order: [['createdAt', 'DESC']],
    limit
  });
};

AuditLog.getUserActivity = function(userId, limit = 50) {
  return this.findAll({
    where: { userId },
    order: [['createdAt', 'DESC']],
    limit
  });
};

AuditLog.getRecentActivity = function(days = 7, limit = 100) {
  const startDate = new Date();
  startDate.setDate(startDate.getDate() - days);

  return this.findAll({
    where: {
      createdAt: {
        [Op.gte]: startDate
      }
    },
    include: ['user'],
    order: [['createdAt', 'DESC']],
    limit
  });
};

module.exports = AuditLog;