const winston = require('winston');
const { createAuditLog } = require('../utils/logger');

// Create Winston logger
const logger = winston.createLogger({
  level: process.env.LOG_LEVEL || 'info',
  format: winston.format.combine(
    winston.format.timestamp(),
    winston.format.errors({ stack: true }),
    winston.format.json()
  ),
  defaultMeta: { service: 'stayspot-backend' },
  transports: [
    new winston.transports.File({ 
      filename: 'logs/error.log', 
      level: 'error',
      maxsize: 5242880, // 5MB
      maxFiles: 5
    }),
    new winston.transports.File({ 
      filename: 'logs/combined.log',
      maxsize: 5242880, // 5MB
      maxFiles: 5
    })
  ]
});

// Add console transport in development
if (process.env.NODE_ENV !== 'production') {
  logger.add(new winston.transports.Console({
    format: winston.format.combine(
      winston.format.colorize(),
      winston.format.simple()
    )
  }));
}

// HTTP request logging middleware
const requestLogger = async (req, res, next) => {
  const start = Date.now();

  // Log request
  logger.info('Incoming request', {
    method: req.method,
    url: req.originalUrl,
    ip: req.ip,
    userAgent: req.get('User-Agent'),
    userId: req.user?.id
  });

  // Capture response
  res.on('finish', async () => {
    const duration = Date.now() - start;
    
    const logData = {
      method: req.method,
      url: req.originalUrl,
      statusCode: res.statusCode,
      duration: `${duration}ms`,
      ip: req.ip,
      userAgent: req.get('User-Agent'),
      userId: req.user?.id,
      contentLength: res.get('Content-Length')
    };

    // Log based on status code
    if (res.statusCode >= 400) {
      logger.warn('Request completed with error', logData);
    } else {
      logger.info('Request completed successfully', logData);
    }

    // Create audit log for significant actions
    if (shouldAudit(req)) {
      await createAuditLog({
        userId: req.user?.id,
        action: getActionFromMethod(req.method),
        entity: getEntityFromUrl(req.originalUrl),
        status: res.statusCode >= 400 ? 'failure' : 'success',
        ipAddress: req.ip,
        userAgent: req.get('User-Agent'),
        metadata: {
          method: req.method,
          url: req.originalUrl,
          statusCode: res.statusCode,
          duration
        }
      });
    }
  });

  next();
};

const shouldAudit = (req) => {
  // Don't audit static files, health checks, etc.
  const excludedPaths = ['/health', '/metrics', '/favicon.ico'];
  if (excludedPaths.includes(req.path)) {
    return false;
  }

  // Only audit specific methods
  const auditedMethods = ['POST', 'PUT', 'PATCH', 'DELETE'];
  return auditedMethods.includes(req.method);
};

const getActionFromMethod = (method) => {
  const actions = {
    'POST': 'CREATE',
    'PUT': 'UPDATE',
    'PATCH': 'UPDATE',
    'DELETE': 'DELETE'
  };
  return actions[method] || 'VIEW';
};

const getEntityFromUrl = (url) => {
  const segments = url.split('/').filter(segment => segment);
  if (segments.length > 1) {
    return segments[1].replace(/s$/, ''); // Convert plural to singular
  }
  return 'System';
};

// Security logging middleware
const securityLogger = (req, res, next) => {
  // Log potential security issues
  const securityChecks = {
    xssAttempt: hasXssAttempt(req),
    sqlInjectionAttempt: hasSqlInjectionAttempt(req),
    pathTraversalAttempt: hasPathTraversalAttempt(req)
  };

  const hasSecurityIssue = Object.values(securityChecks).some(check => check);

  if (hasSecurityIssue) {
    logger.warn('Potential security issue detected', {
      ...securityChecks,
      method: req.method,
      url: req.originalUrl,
      ip: req.ip,
      userAgent: req.get('User-Agent')
    });

    // Create security audit log
    createAuditLog({
      userId: req.user?.id,
      action: 'SECURITY_ALERT',
      entity: 'System',
      status: 'failure',
      ipAddress: req.ip,
      userAgent: req.get('User-Agent'),
      metadata: securityChecks
    }).catch(error => {
      logger.error('Failed to create security audit log', { error: error.message });
    });
  }

  next();
};

const hasXssAttempt = (req) => {
  const xssPatterns = [/<script/i, /javascript:/i, /onload=/i, /onerror=/i];
  return checkPatternsInRequest(req, xssPatterns);
};

const hasSqlInjectionAttempt = (req) => {
  const sqlPatterns = [/' OR '1'='1/i, /UNION SELECT/i, /DROP TABLE/i, /INSERT INTO/i];
  return checkPatternsInRequest(req, sqlPatterns);
};

const hasPathTraversalAttempt = (req) => {
  const pathPatterns = [/\.\.\//i, /\.\.\\/i, /etc\/passwd/i, /win\.ini/i];
  return checkPatternsInRequest(req, pathPatterns);
};

const checkPatternsInRequest = (req, patterns) => {
  const checkValue = (value) => {
    if (typeof value === 'string') {
      return patterns.some(pattern => pattern.test(value));
    }
    return false;
  };

  // Check query parameters
  if (Object.values(req.query).some(checkValue)) return true;
  
  // Check body parameters
  if (req.body && Object.values(req.body).some(checkValue)) return true;
  
  // Check URL
  if (checkValue(req.originalUrl)) return true;
  
  return false;
};

module.exports = {
  logger,
  requestLogger,
  securityLogger
};