const { User, Property, Lease, Payment, AuditLog } = require('../models');
const { Op } = require('sequelize');
const authConfig = require('../config/auth');

const userController = {
  // Get all users (admin only)
  getAllUsers: async (req, res) => {
    try {
      const { page = 1, limit = 10, search, role, isActive } = req.query;
      const offset = (page - 1) * limit;

      const where = {};
      if (search) {
        where[Op.or] = [
          { firstName: { [Op.like]: `%${search}%` } },
          { lastName: { [Op.like]: `%${search}%` } },
          { email: { [Op.like]: `%${search}%` } }
        ];
      }
      if (role) where.role = role;
      if (isActive !== undefined) where.isActive = isActive === 'true';

      const users = await User.findAndCountAll({
        where,
        attributes: { exclude: ['password'] },
        limit: parseInt(limit),
        offset: parseInt(offset),
        order: [['createdAt', 'DESC']]
      });

      res.json({
        success: true,
        data: {
          users: users.rows,
          pagination: {
            page: parseInt(page),
            limit: parseInt(limit),
            total: users.count,
            pages: Math.ceil(users.count / limit)
          }
        }
      });
    } catch (error) {
      console.error('Get users error:', error);
      res.status(500).json({
        success: false,
        message: 'Error fetching users'
      });
    }
  },

  // Get user by ID
  getUserById: async (req, res) => {
    try {
      const { id } = req.params;

      const user = await User.findByPk(id, {
        attributes: { exclude: ['password'] },
        include: [
          {
            model: Property,
            as: 'properties',
            attributes: ['id', 'title', 'address', 'status']
          },
          {
            model: Lease,
            as: 'tenantLeases',
            include: [{
              model: Property,
              attributes: ['id', 'title', 'address']
            }]
          }
        ]
      });

      if (!user) {
        return res.status(404).json({
          success: false,
          message: 'User not found'
        });
      }

      res.json({
        success: true,
        data: { user }
      });
    } catch (error) {
      console.error('Get user error:', error);
      res.status(500).json({
        success: false,
        message: 'Error fetching user'
      });
    }
  },

  // Update user
  updateUser: async (req, res) => {
    try {
      const { id } = req.params;
      const updateData = { ...req.body };

      const user = await User.findByPk(id);
      if (!user) {
        return res.status(404).json({
          success: false,
          message: 'User not found'
        });
      }

      // Check permissions
      if (req.user.id !== parseInt(id) && !req.user.role.includes('admin')) {
        return res.status(403).json({
          success: false,
          message: 'Not authorized to update this user'
        });
      }

      // Admins can update role and active status
      if (!req.user.role.includes('admin')) {
        delete updateData.role;
        delete updateData.isActive;
      }

      await user.update(updateData);

      // Log the action
      await AuditLog.logAction({
        action: 'user_updated',
        entityType: 'user',
        entityId: user.id,
        userId: req.user.id,
        affectedUserId: user.id,
        description: `User ${user.email} updated by ${req.user.email}`,
        ipAddress: req.ip,
        userAgent: req.get('User-Agent')
      });

      const updatedUser = await User.findByPk(id, {
        attributes: { exclude: ['password'] }
      });

      res.json({
        success: true,
        message: 'User updated successfully',
        data: { user: updatedUser }
      });
    } catch (error) {
      console.error('Update user error:', error);
      res.status(500).json({
        success: false,
        message: 'Error updating user'
      });
    }
  },

  // Delete user
  deleteUser: async (req, res) => {
    try {
      const { id } = req.params;

      const user = await User.findByPk(id);
      if (!user) {
        return res.status(404).json({
          success: false,
          message: 'User not found'
        });
      }

      // Prevent self-deletion
      if (req.user.id === parseInt(id)) {
        return res.status(400).json({
          success: false,
          message: 'Cannot delete your own account'
        });
      }

      // Check if user has properties or active leases
      const propertyCount = await Property.count({ where: { landlordId: id } });
      const activeLeaseCount = await Lease.count({ 
        where: { 
          [Op.or]: [
            { tenantId: id, status: 'active' },
            { landlordId: id, status: 'active' }
          ]
        }
      });

      if (propertyCount > 0 || activeLeaseCount > 0) {
        return res.status(400).json({
          success: false,
          message: 'Cannot delete user with active properties or leases'
        });
      }

      await user.destroy();

      // Log the action
      await AuditLog.logAction({
        action: 'user_deleted',
        entityType: 'user',
        entityId: user.id,
        userId: req.user.id,
        affectedUserId: user.id,
        description: `User ${user.email} deleted by ${req.user.email}`,
        ipAddress: req.ip,
        userAgent: req.get('User-Agent')
      });

      res.json({
        success: true,
        message: 'User deleted successfully'
      });
    } catch (error) {
      console.error('Delete user error:', error);
      res.status(500).json({
        success: false,
        message: 'Error deleting user'
      });
    }
  },

  // Get user statistics
  getUserStats: async (req, res) => {
    try {
      const { id } = req.params;
      const userId = id || req.user.id;

      const user = await User.findByPk(userId);
      if (!user) {
        return res.status(404).json({
          success: false,
          message: 'User not found'
        });
      }

      let stats = {};

      if (user.role === authConfig.roles.LANDLORD) {
        const propertyCount = await Property.count({ where: { landlordId: userId } });
        const activeLeaseCount = await Lease.count({ 
          where: { 
            landlordId: userId, 
            status: 'active' 
          } 
        });
        const totalRevenue = await Payment.sum('amount', {
          where: {
            landlordId: userId,
            status: 'paid'
          }
        });

        stats = {
          propertyCount,
          activeLeaseCount,
          totalRevenue: totalRevenue || 0,
          vacantProperties: propertyCount - activeLeaseCount
        };
      } else if (user.role === authConfig.roles.TENANT) {
        const activeLeaseCount = await Lease.count({ 
          where: { 
            tenantId: userId, 
            status: 'active' 
          } 
        });
        const totalPaid = await Payment.sum('amount', {
          where: {
            tenantId: userId,
            status: 'paid'
          }
        });
        const maintenanceRequests = await Maintenance.count({
          where: { tenantId: userId }
        });

        stats = {
          activeLeaseCount,
          totalPaid: totalPaid || 0,
          maintenanceRequests
        };
      }

      res.json({
        success: true,
        data: { stats }
      });
    } catch (error) {
      console.error('Get user stats error:', error);
      res.status(500).json({
        success: false,
        message: 'Error fetching user statistics'
      });
    }
  },

  // Upload user avatar
  uploadAvatar: async (req, res) => {
    try {
      const user = await User.findByPk(req.user.id);
      if (!user) {
        return res.status(404).json({
          success: false,
          message: 'User not found'
        });
      }

      if (!req.uploadedFiles || req.uploadedFiles.length === 0) {
        return res.status(400).json({
          success: false,
          message: 'No file uploaded'
        });
      }

      const avatar = req.uploadedFiles[0];
      await user.update({ avatar: avatar.url });

      res.json({
        success: true,
        message: 'Avatar uploaded successfully',
        data: { avatar: avatar.url }
      });
    } catch (error) {
      console.error('Upload avatar error:', error);
      res.status(500).json({
        success: false,
        message: 'Error uploading avatar'
      });
    }
  },

  // Get user activity
  getUserActivity: async (req, res) => {
    try {
      const { id } = req.params;
      const userId = id || req.user.id;
      const { page = 1, limit = 20 } = req.query;
      const offset = (page - 1) * limit;

      const activity = await AuditLog.findAndCountAll({
        where: { userId },
        include: ['user'],
        order: [['createdAt', 'DESC']],
        limit: parseInt(limit),
        offset: parseInt(offset)
      });

      res.json({
        success: true,
        data: {
          activity: activity.rows,
          pagination: {
            page: parseInt(page),
            limit: parseInt(limit),
            total: activity.count,
            pages: Math.ceil(activity.count / limit)
          }
        }
      });
    } catch (error) {
      console.error('Get user activity error:', error);
      res.status(500).json({
        success: false,
        message: 'Error fetching user activity'
      });
    }
  }
};

module.exports = userController;