const { Notification, User } = require('../models');
const { Op } = require('sequelize');

const notificationController = {
  // Get user notifications
  getUserNotifications: async (req, res) => {
    try {
      const { page = 1, limit = 20, type, isRead } = req.query;
      const offset = (page - 1) * limit;

      const where = { userId: req.user.id };
      if (type) where.type = type;
      if (isRead !== undefined) where.isRead = isRead === 'true';

      const notifications = await Notification.findAndCountAll({
        where,
        order: [['createdAt', 'DESC']],
        limit: parseInt(limit),
        offset: parseInt(offset)
      });

      res.json({
        success: true,
        data: {
          notifications: notifications.rows,
          pagination: {
            page: parseInt(page),
            limit: parseInt(limit),
            total: notifications.count,
            pages: Math.ceil(notifications.count / limit)
          }
        }
      });
    } catch (error) {
      console.error('Get notifications error:', error);
      res.status(500).json({
        success: false,
        message: 'Error fetching notifications'
      });
    }
  },

  // Get notification by ID
  getNotificationById: async (req, res) => {
    try {
      const { id } = req.params;

      const notification = await Notification.findOne({
        where: {
          id,
          userId: req.user.id
        }
      });

      if (!notification) {
        return res.status(404).json({
          success: false,
          message: 'Notification not found'
        });
      }

      res.json({
        success: true,
        data: { notification }
      });
    } catch (error) {
      console.error('Get notification error:', error);
      res.status(500).json({
        success: false,
        message: 'Error fetching notification'
      });
    }
  },

  // Mark notification as read
  markAsRead: async (req, res) => {
    try {
      const { id } = req.params;

      const notification = await Notification.findOne({
        where: {
          id,
          userId: req.user.id
        }
      });

      if (!notification) {
        return res.status(404).json({
          success: false,
          message: 'Notification not found'
        });
      }

      await notification.markAsRead();

      res.json({
        success: true,
        message: 'Notification marked as read',
        data: { notification }
      });
    } catch (error) {
      console.error('Mark notification as read error:', error);
      res.status(500).json({
        success: false,
        message: 'Error marking notification as read'
      });
    }
  },

  // Mark all notifications as read
  markAllAsRead: async (req, res) => {
    try {
      await Notification.update(
        { isRead: true },
        {
          where: {
            userId: req.user.id,
            isRead: false
          }
        }
      );

      res.json({
        success: true,
        message: 'All notifications marked as read'
      });
    } catch (error) {
      console.error('Mark all notifications as read error:', error);
      res.status(500).json({
        success: false,
        message: 'Error marking all notifications as read'
      });
    }
  },

  // Delete notification
  deleteNotification: async (req, res) => {
    try {
      const { id } = req.params;

      const notification = await Notification.findOne({
        where: {
          id,
          userId: req.user.id
        }
      });

      if (!notification) {
        return res.status(404).json({
          success: false,
          message: 'Notification not found'
        });
      }

      await notification.destroy();

      res.json({
        success: true,
        message: 'Notification deleted successfully'
      });
    } catch (error) {
      console.error('Delete notification error:', error);
      res.status(500).json({
        success: false,
        message: 'Error deleting notification'
      });
    }
  },

  // Clear all notifications
  clearAllNotifications: async (req, res) => {
    try {
      await Notification.destroy({
        where: {
          userId: req.user.id
        }
      });

      res.json({
        success: true,
        message: 'All notifications cleared'
      });
    } catch (error) {
      console.error('Clear all notifications error:', error);
      res.status(500).json({
        success: false,
        message: 'Error clearing all notifications'
      });
    }
  },

  // Get unread count
  getUnreadCount: async (req, res) => {
    try {
      const count = await Notification.count({
        where: {
          userId: req.user.id,
          isRead: false
        }
      });

      res.json({
        success: true,
        data: { unreadCount: count }
      });
    } catch (error) {
      console.error('Get unread count error:', error);
      res.status(500).json({
        success: false,
        message: 'Error fetching unread count'
      });
    }
  },

  // Get notifications by type
  getNotificationsByType: async (req, res) => {
    try {
      const { type } = req.params;
      const { page = 1, limit = 20 } = req.query;
      const offset = (page - 1) * limit;

      const notifications = await Notification.findAndCountAll({
        where: {
          userId: req.user.id,
          type
        },
        order: [['createdAt', 'DESC']],
        limit: parseInt(limit),
        offset: parseInt(offset)
      });

      res.json({
        success: true,
        data: {
          notifications: notifications.rows,
          pagination: {
            page: parseInt(page),
            limit: parseInt(limit),
            total: notifications.count,
            pages: Math.ceil(notifications.count / limit)
          }
        }
      });
    } catch (error) {
      console.error('Get notifications by type error:', error);
      res.status(500).json({
        success: false,
        message: 'Error fetching notifications by type'
      });
    }
  }
};

module.exports = notificationController;