const { Lease, Property, User, Payment, Document } = require('../models');
const { Op } = require('sequelize');
const moment = require('moment');

const leaseController = {
  // Get all leases
  getAllLeases: async (req, res) => {
    try {
      const { page = 1, limit = 10, status, propertyId, tenantId, landlordId } = req.query;
      const offset = (page - 1) * limit;

      const where = {};
      if (status) where.status = status;
      if (propertyId) where.propertyId = propertyId;
      if (tenantId) where.tenantId = tenantId;
      if (landlordId) where.landlordId = landlordId;

      // Filter based on user role
      if (req.user.role === 'tenant') {
        where.tenantId = req.user.id;
      } else if (req.user.role === 'landlord') {
        where.landlordId = req.user.id;
      }

      const leases = await Lease.findAndCountAll({
        where,
        include: [
          {
            model: Property,
            as: 'property',
            attributes: ['id', 'title', 'address', 'images']
          },
          {
            model: User,
            as: 'tenant',
            attributes: ['id', 'firstName', 'lastName', 'email', 'phone']
          },
          {
            model: User,
            as: 'landlord',
            attributes: ['id', 'firstName', 'lastName', 'email', 'phone']
          }
        ],
        limit: parseInt(limit),
        offset: parseInt(offset),
        order: [['createdAt', 'DESC']]
      });

      res.json({
        success: true,
        data: {
          leases: leases.rows,
          pagination: {
            page: parseInt(page),
            limit: parseInt(limit),
            total: leases.count,
            pages: Math.ceil(leases.count / limit)
          }
        }
      });
    } catch (error) {
      console.error('Get leases error:', error);
      res.status(500).json({
        success: false,
        message: 'Error fetching leases'
      });
    }
  },

  // Get lease by ID
  getLeaseById: async (req, res) => {
    try {
      const { id } = req.params;

      const lease = await Lease.findByPk(id, {
        include: [
          {
            model: Property,
            as: 'property',
            include: ['landlord']
          },
          {
            model: User,
            as: 'tenant',
            attributes: { exclude: ['password'] }
          },
          {
            model: User,
            as: 'landlord',
            attributes: { exclude: ['password'] }
          },
          {
            model: Payment,
            as: 'payments',
            order: [['dueDate', 'DESC']]
          },
          {
            model: Document,
            as: 'documents',
            where: { documentType: 'lease_agreement' },
            required: false
          }
        ]
      });

      if (!lease) {
        return res.status(404).json({
          success: false,
          message: 'Lease not found'
        });
      }

      // Check permissions
      if (lease.tenantId !== req.user.id && lease.landlordId !== req.user.id && !req.user.role.includes('admin')) {
        return res.status(403).json({
          success: false,
          message: 'Not authorized to view this lease'
        });
      }

      res.json({
        success: true,
        data: { lease }
      });
    } catch (error) {
      console.error('Get lease error:', error);
      res.status(500).json({
        success: false,
        message: 'Error fetching lease'
      });
    }
  },

  // Create new lease
  createLease: async (req, res) => {
    try {
      const leaseData = {
        ...req.body,
        landlordId: req.user.role === 'landlord' ? req.user.id : req.body.landlordId
      };

      // Verify property exists and is available
      const property = await Property.findByPk(leaseData.propertyId);
      if (!property) {
        return res.status(404).json({
          success: false,
          message: 'Property not found'
        });
      }

      if (property.status !== 'available') {
        return res.status(400).json({
          success: false,
          message: 'Property is not available for lease'
        });
      }

      // Verify tenant exists
      const tenant = await User.findByPk(leaseData.tenantId);
      if (!tenant) {
        return res.status(404).json({
          success: false,
          message: 'Tenant not found'
        });
      }

      // Check for overlapping leases
      const overlappingLease = await Lease.findOne({
        where: {
          propertyId: leaseData.propertyId,
          status: 'active',
          [Op.or]: [
            {
              startDate: { [Op.lte]: leaseData.endDate },
              endDate: { [Op.gte]: leaseData.startDate }
            }
          ]
        }
      });

      if (overlappingLease) {
        return res.status(400).json({
          success: false,
          message: 'Property already has an active lease for this period'
        });
      }

      const lease = await Lease.create(leaseData);

      // Update property status
      await property.update({ status: 'occupied' });

      // Create initial payment records
      await createPaymentSchedule(lease);

      const createdLease = await Lease.findByPk(lease.id, {
        include: ['property', 'tenant', 'landlord']
      });

      res.status(201).json({
        success: true,
        message: 'Lease created successfully',
        data: { lease: createdLease }
      });
    } catch (error) {
      console.error('Create lease error:', error);
      res.status(500).json({
        success: false,
        message: 'Error creating lease'
      });
    }
  },

  // Update lease
  updateLease: async (req, res) => {
    try {
      const { id } = req.params;
      const updateData = req.body;

      const lease = await Lease.findByPk(id);
      if (!lease) {
        return res.status(404).json({
          success: false,
          message: 'Lease not found'
        });
      }

      // Check permissions
      if (lease.landlordId !== req.user.id && !req.user.role.includes('admin')) {
        return res.status(403).json({
          success: false,
          message: 'Not authorized to update this lease'
        });
      }

      // Prevent updating active lease dates
      if (lease.status === 'active' && (updateData.startDate || updateData.endDate)) {
        return res.status(400).json({
          success: false,
          message: 'Cannot update dates of an active lease'
        });
      }

      await lease.update(updateData);

      const updatedLease = await Lease.findByPk(id, {
        include: ['property', 'tenant', 'landlord']
      });

      res.json({
        success: true,
        message: 'Lease updated successfully',
        data: { lease: updatedLease }
      });
    } catch (error) {
      console.error('Update lease error:', error);
      res.status(500).json({
        success: false,
        message: 'Error updating lease'
      });
    }
  },

  // Terminate lease
  terminateLease: async (req, res) => {
    try {
      const { id } = req.params;
      const { terminationReason } = req.body;

      const lease = await Lease.findByPk(id);
      if (!lease) {
        return res.status(404).json({
          success: false,
          message: 'Lease not found'
        });
      }

      if (lease.status !== 'active') {
        return res.status(400).json({
          success: false,
          message: 'Only active leases can be terminated'
        });
      }

      // Check permissions
      if (lease.landlordId !== req.user.id && !req.user.role.includes('admin')) {
        return res.status(403).json({
          success: false,
          message: 'Not authorized to terminate this lease'
        });
      }

      await lease.update({
        status: 'terminated',
        terminatedAt: new Date(),
        specialConditions: terminationReason ? 
          `${lease.specialConditions || ''}\nTermination Reason: ${terminationReason}`.trim() 
          : lease.specialConditions
      });

      // Update property status
      const property = await Property.findByPk(lease.propertyId);
      await property.update({ status: 'available' });

      res.json({
        success: true,
        message: 'Lease terminated successfully',
        data: { lease }
      });
    } catch (error) {
      console.error('Terminate lease error:', error);
      res.status(500).json({
        success: false,
        message: 'Error terminating lease'
      });
    }
  },

  // Renew lease
  renewLease: async (req, res) => {
    try {
      const { id } = req.params;
      const { newEndDate, newMonthlyRent } = req.body;

      const lease = await Lease.findByPk(id);
      if (!lease) {
        return res.status(404).json({
          success: false,
          message: 'Lease not found'
        });
      }

      if (lease.status !== 'active') {
        return res.status(400).json({
          success: false,
          message: 'Only active leases can be renewed'
        });
      }

      // Check permissions
      if (lease.landlordId !== req.user.id && !req.user.role.includes('admin')) {
        return res.status(403).json({
          success: false,
          message: 'Not authorized to renew this lease'
        });
      }

      const renewalData = {
        endDate: newEndDate,
        monthlyRent: newMonthlyRent || lease.monthlyRent
      };

      await lease.update(renewalData);

      // Create new payment schedule for renewal period
      await createPaymentSchedule(lease, new Date(lease.endDate), new Date(newEndDate));

      res.json({
        success: true,
        message: 'Lease renewed successfully',
        data: { lease }
      });
    } catch (error) {
      console.error('Renew lease error:', error);
      res.status(500).json({
        success: false,
        message: 'Error renewing lease'
      });
    }
  },

  // Get lease payments
  getLeasePayments: async (req, res) => {
    try {
      const { id } = req.params;
      const { page = 1, limit = 12, status } = req.query;
      const offset = (page - 1) * limit;

      const lease = await Lease.findByPk(id);
      if (!lease) {
        return res.status(404).json({
          success: false,
          message: 'Lease not found'
        });
      }

      // Check permissions
      if (lease.tenantId !== req.user.id && lease.landlordId !== req.user.id && !req.user.role.includes('admin')) {
        return res.status(403).json({
          success: false,
          message: 'Not authorized to view these payments'
        });
      }

      const where = { leaseId: id };
      if (status) where.status = status;

      const payments = await Payment.findAndCountAll({
        where,
        order: [['dueDate', 'DESC']],
        limit: parseInt(limit),
        offset: parseInt(offset)
      });

      res.json({
        success: true,
        data: {
          payments: payments.rows,
          pagination: {
            page: parseInt(page),
            limit: parseInt(limit),
            total: payments.count,
            pages: Math.ceil(payments.count / limit)
          }
        }
      });
    } catch (error) {
      console.error('Get lease payments error:', error);
      res.status(500).json({
        success: false,
        message: 'Error fetching lease payments'
      });
    }
  }
};

// Helper function to create payment schedule
const createPaymentSchedule = async (lease, startDate = null, endDate = null) => {
  const start = startDate || new Date(lease.startDate);
  const end = endDate || new Date(lease.endDate);
  const monthlyRent = lease.monthlyRent;

  const payments = [];
  let currentDate = new Date(start);

  while (currentDate < end) {
    const dueDate = new Date(currentDate);
    dueDate.setMonth(dueDate.getMonth() + 1);
    
    if (dueDate > end) break;

    payments.push({
      amount: monthlyRent,
      dueDate: dueDate,
      status: 'pending',
      leaseId: lease.id,
      tenantId: lease.tenantId,
      landlordId: lease.landlordId,
      propertyId: lease.propertyId,
      description: `Rent for ${dueDate.toLocaleDateString('en-US', { month: 'long', year: 'numeric' })}`
    });

    currentDate = new Date(dueDate);
  }

  if (payments.length > 0) {
    await Payment.bulkCreate(payments);
  }
};

module.exports = leaseController;